package io.intino.sumus.queries.temporalrecord;

import io.intino.sumus.TimeStamp;
import io.intino.sumus.graph.AbstractCategorization;
import io.intino.sumus.graph.SumusGraph;
import io.intino.sumus.graph.TemporalRecord;
import io.intino.tara.magritte.Concept;
import io.intino.tara.magritte.Graph;

import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Objects;

import static io.intino.sumus.datawarehouse.store.PathBuilder.temporalRecordPath;
import static java.util.stream.Collectors.toList;

public class QueryExecutor {

	private final Query query;
	private final List<AbstractCategorization.RecordTagger> taggers;
	private final QueryResult result;

	private QueryExecutor(Query query) {
		this.query = query;
		this.taggers = query.nameSpace().graph().categorizationList().stream()
				.map(t -> t.recordTaggerOf(query.temporalRecord()))
				.filter(Objects::nonNull)
				.collect(toList());
		this.result = new QueryResult(query);
	}

	public static QueryResult execute(Query query) {
		QueryExecutor executor = new QueryExecutor(query);
		executor.execute();
		return executor.result;
	}

	private void execute() {
		SumusGraph graph = query.nameSpace().graph();
		List<Concept> concepts = subConceptsOf(graph, query.temporalRecord());
		for (TimeStamp timeStamp : query.timeStamps()) {
			Graph clone = query.nameSpace().graph().core$().clone();
			loadStashes(clone, concepts, timeStamp);
			clone.rootList(query.temporalRecord().layerClass()).stream()
					.map(e -> e.a$(TemporalRecord.class))
					.filter(this::isInFilter)
					.filter(this::hasCondition)
					.forEach(result::register);
		}
	}

	private List<Concept> subConceptsOf(SumusGraph graph, Concept concept) {
		return graph.core$().conceptList().stream().filter(c -> c.is(concept.name())).collect(toList());
	}

	private void loadStashes(Graph graph, List<Concept> concepts, TimeStamp timeStamp) {
		concepts.forEach(tr -> graph.loadStashes(temporalRecordPath(query.nameSpace(), tr, timeStamp)));
	}

	private boolean hasCondition(TemporalRecord temporalRecord) {
		String condition = query.condition();
		if (condition == null || condition.isEmpty()) return true;
		final List<String> keys = Arrays.asList(condition.toLowerCase().split(" "));
		List<String> words = temporalRecord.core$().variables().values().stream().flatMap(Collection::stream).filter(s -> s instanceof String).map(s -> (String) s).collect(toList());
		return keys.stream().filter(key -> words.stream().filter(word -> word.toLowerCase().contains(key)).count() > 0).count() == keys.size();
	}

	private boolean isInFilter(TemporalRecord temporalRecord) {
		return query.filter() == null ||
				query.filter().contains(tagsOf(temporalRecord));
	}

	private Collection<String> tagsOf(TemporalRecord temporalRecord) {
		return temporalRecord.graph().categorizationList().stream()
				.map(c -> c.recordTaggerOf(temporalRecord))
				.filter(Objects::nonNull)
				.map(c -> c.tag(temporalRecord))
				.flatMap(Collection::stream)
				.collect(toList());
	}

}
