package io.intino.sumus.queries.digest;

import io.intino.sumus.TimeStamp;
import io.intino.sumus.datawarehouse.store.Bucket;
import io.intino.sumus.datawarehouse.store.Digest;
import io.intino.sumus.graph.AbstractCategorization.DigestTagger;
import io.intino.sumus.graph.Cube;
import io.intino.sumus.graph.NameSpace;

import java.util.*;
import java.util.concurrent.ConcurrentHashMap;

import static java.util.stream.Collectors.toList;

public class QueryExecutor {

	private final Query query;
	private final QueryResult result;
	private final Map<Cube, List<DigestTagger>> taggers = new ConcurrentHashMap<>();

	private QueryExecutor(Query query) {
		this.query = query.commit();
		this.result = new QueryResult(query);
	}

	public static QueryResult execute(Query query) {
		QueryExecutor executor = new QueryExecutor(query);
		executor.execute();
		return executor.result;
	}

	private void execute() {

		for (NameSpace nameSpace : query.nameSpaces()) processNamespaces(nameSpace);
	}

	private void processNamespaces(NameSpace nameSpace) {
		List<Query.SubQuery> queries = query.subQueries().stream().filter(s -> s.nameSpace.equals(nameSpace)).collect(toList());
		queryTimeStamps(queries).parallelStream().forEach(i -> processNamespace(i, queries));
	}

	private void processNamespace(TimeStamp stamp, Collection<Query.SubQuery> queries) {
		Collection<Query.SubQuery> availableQueries = availableQueries(stamp, queries);
		Map<Cube, BucketWithDrills> bucketMap = bucketByCubeType(stamp, availableQueries);
		for (Query.SubQuery query : availableQueries) {
			BucketWithDrills bucket = bucketMap.get(query.cube());
			if (bucket == null) continue;
			if (bucket.drilledDigests.isEmpty())
				result.register(query, stamp, query.formula.calculate(bucket.filteredDigests));
			else
				bucket.drilledDigests.forEach((drill, value) -> result.register(query, drill, stamp, query.formula.calculate(value)));
		}
	}

	private Collection<Query.SubQuery> availableQueries(TimeStamp stamp, Collection<Query.SubQuery> queries) {
		if (query.timeStamps().contains(stamp)) return queries;
		List<Query.SubQuery> subQueries = new ArrayList<>();
		for (Query.SubQuery subQuery : queries) if (subQuery.ts.contains(stamp)) subQueries.add(subQuery);
		return subQueries;
	}

	private Map<io.intino.sumus.queries.Drill, List<Digest>> groupDigestsByDrill(List<DigestWithTags> digests, List<io.intino.sumus.queries.Drill> temporalDrills) {
		Map<io.intino.sumus.queries.Drill, List<Digest>> digestsByDrill = new HashMap<>();
		if (!query.drills().isEmpty()) groupDigestsByDrill(digests, digestsByDrill);
		if (temporalDrills != null) groupDigestsByDrill(digests, digestsByDrill, temporalDrills);
		return digestsByDrill;
	}

	private synchronized void groupDigestsByDrill(List<DigestWithTags> digests, Map<io.intino.sumus.queries.Drill, List<Digest>> digestsByDrill) {
		query.drills().forEach(d -> digestsByDrill.put(d, new ArrayList<>()));
		for (DigestWithTags d : digests)
			query.drills().stream()
					.filter(drill -> drill.contains(d.tags))
					.forEach(drill -> digestsByDrill.get(drill).add(d.digest));
	}

	private void groupDigestsByDrill(List<DigestWithTags> digests, Map<io.intino.sumus.queries.Drill, List<Digest>> digestsByDrill, List<io.intino.sumus.queries.Drill> temporalDrills) {
		temporalDrills.forEach(d -> digestsByDrill.put(d, new ArrayList<>()));
		for (DigestWithTags d : digests)
			temporalDrills.stream()
					.filter(t -> t.contains(d.digest.ts()))
					.forEach(t -> digestsByDrill.get(t).add(d.digest));
	}

	private Collection<TimeStamp> queryTimeStamps(List<Query.SubQuery> queries) {
		Set<TimeStamp> stamps = new LinkedHashSet<>();
		queries.forEach(q -> stamps.addAll(q.ts));
		return stamps;
	}

	private List<DigestWithTags> filter(List<DigestWithTags> digests) {
		if (query.filter() == null) return digests;
		List<DigestWithTags> result = new ArrayList<>();
		for (DigestWithTags digest : digests) if (query.filter().contains(digest.tags)) result.add(digest);
		return result;
	}

	private Map<Cube, BucketWithDrills> bucketByCubeType(TimeStamp stamp, Collection<Query.SubQuery> queries) {
		Map<Cube, BucketWithDrills> cubes = new HashMap<>();
		for (Cube cube : cubes(queries)) {
			Bucket bucket = new Bucket(cube, queries.iterator().next().nameSpace, stamp);
			if (!bucket.exists()) continue;
			cubes.put(cube, new BucketWithDrills(bucket));
		}
		return cubes;
	}

	private Set<Cube> cubes(Collection<Query.SubQuery> queries) {
		Set<Cube> concepts = new HashSet<>();
		for (Query.SubQuery query : queries) concepts.add(query.cube());
		return concepts;
	}

	private List<DigestWithTags> withTags(Cube cube, List<Digest> digests) {
		List<DigestWithTags> result = digests.stream().map(DigestWithTags::new).collect(toList());
		if (query.filter() == null && query.drills().isEmpty()) return result;
		List<DigestTagger> taggers = taggersOf(cube);
		result.forEach(d -> taggers.forEach(t -> d.addTags(t.tag(d.digest))));
		return result;
	}

	private List<DigestTagger> taggersOf(Cube cube) {
		return cube.graph().categorizationList().stream()
				.map(c -> c.digestTagger(cube))
				.filter(Objects::nonNull)
				.collect(toList());

	}

	class BucketWithDrills {
		List<Digest> filteredDigests;
		Map<io.intino.sumus.queries.Drill, List<Digest>> drilledDigests;

		BucketWithDrills(Bucket bucket) {
			List<DigestWithTags> digestsWithTags = filter(withTags(bucket.cube(), bucket.digests()));
			filteredDigests = digestsWithTags.stream().map(d -> d.digest).collect(toList());
			result.register(filteredDigests);
			drilledDigests = groupDigestsByDrill(digestsWithTags, query.drills(bucket.timeStamp()));
		}
	}

	class DigestWithTags {

		Digest digest;
		Set<String> tags = new HashSet<>();

		DigestWithTags(Digest digest) {
			this.digest = digest;
		}

		void addTags(List<String> tags) {
			this.tags.addAll(tags);
		}
	}

}
