package io.intino.sumus.helpers;

import io.intino.konos.alexandria.activity.services.push.User;
import io.intino.sumus.box.SumusBox;
import io.intino.sumus.graph.Entity;
import io.intino.sumus.queries.EntityQuery;
import io.intino.sumus.queries.Scope;
import io.intino.tara.magritte.Concept;
import io.intino.tara.magritte.Layer;
import io.intino.tara.magritte.Node;

import java.text.Normalizer;
import java.util.*;

import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;

public class EntityHelper extends Helper {
    private final SumusBox box;

    public EntityHelper(SumusBox box) {
        this.box = box;
    }

    public List<Entity> entities(EntityQuery query) {
        Concept entity = query.entity();
        Optional<Scope> scope = query.scope();
        if (!scope.isPresent()) return Collections.emptyList();
        List<? extends Layer> layers = box.graph().core$().find(entity.layerClass());
        List<Set<String>> lists = scope.get().tagsMap().keySet().stream()
                .map(s -> recordsOfCategorization(s, layers, scope.get().tagsMap().get(s)))
                .collect(toList());
        return layers.stream()
                .filter(l -> lists.stream().allMatch(list -> list.contains(l.core$().id())))
                .map(l -> l.a$(Entity.class))
                .filter(e -> hasCondition(e, query.condition()))
                .collect(toList());
    }

    private Set<String> recordsOfCategorization(String categorization, List<? extends Layer> layers, List<String> tags) {
        if (layers.isEmpty()) return Collections.emptySet();
        return box.graph().categorizationList(c -> c.name$().equals(categorization)).findFirst().get()
                .categorizedLayersMap(layers).entrySet().stream()
                .filter(e -> tags.contains(e.getKey()))
                .map(Map.Entry::getValue)
                .flatMap(Collection::stream)
                .map(l -> l.core$().id())
                .collect(toSet());
    }

    public Entity entity(String id, User user) {
        Node node = box.graph().core$().clone().load(id, false);
        if (node == null) return null;
        return node.as(Entity.class);
    }

    private boolean hasCondition(Layer entity, String condition) {
        if (condition == null || condition.isEmpty()) return true;
        final List<String> keys = Arrays.asList(clean(condition).split(" "));
        List<String> words = words(entity);
        return keys.stream().filter(key -> words.stream().filter(word -> word.contains(key)).count() > 0).count() == keys.size();
    }

    private List<String> words(Layer entity) {
        return words(entity, 0);
    }

    private List<String> words(Layer entity, int depth) {
        if (depth == 2) return Collections.emptyList();

        List<?> values = entity.core$().variables().values().stream().flatMap(Collection::stream).collect(toList());
        List<String> words = values.stream().filter(v -> v instanceof String).map(v -> clean((String)v)).collect(toList());
        values.stream().filter(v -> v instanceof Layer).map(v -> (Layer) v).forEach(e -> words.addAll(words(e, depth + 1)));

        return words;
    }

    private String clean(String value) {
        return Normalizer.normalize(value.toLowerCase(), Normalizer.Form.NFD).replaceAll("[^\\p{ASCII}]", "");
    }
}