package io.intino.sumus.box.displays.builders;

import com.google.gson.Gson;
import io.intino.sumus.box.schemas.Property;
import io.intino.sumus.box.schemas.RecordItem;
import io.intino.sumus.box.schemas.RecordItemBlock;
import io.intino.sumus.box.schemas.RecordItemStamp;
import io.intino.sumus.graph.Event;
import io.intino.sumus.graph.Mold;
import io.intino.sumus.graph.Mold.Block.*;
import io.intino.sumus.graph.Record;
import io.intino.sumus.graph.functions.CatalogStampRecordLinks;
import io.intino.sumus.graph.functions.Tree;
import io.intino.sumus.graph.rules.TimeScale;

import java.net.URL;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Base64;
import java.util.List;
import java.util.UUID;

import static io.intino.konos.server.activity.Asset.toResource;
import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toList;

public class RecordItemBuilder {

    public static RecordItem build(Record record, RecordItemBuilderProvider provider, URL baseAssetUrl) {
        String id = record != null ? record.core$().id() : UUID.randomUUID().toString();

        return new RecordItem().name(new String(Base64.getEncoder().encode(id.getBytes())))
                                .group(group(record, provider.scale()))
                                .label(label(record, provider))
                                .recordItemBlockList(recordItemBlockList(record, provider, baseAssetUrl))
                                .recordItemStampList(recordItemStampList(record, provider, baseAssetUrl));
    }

    public static List<RecordItem> buildList(List<Record> recordList, RecordItemBuilderProvider provider, URL baseAssetUrl) {
        return recordList.stream().map(record -> RecordItemBuilder.build(record, provider, baseAssetUrl)).collect(toList());
    }

    private static String label(Record record, RecordItemBuilderProvider provider) {
        String defaultLabel = record != null ? record.name$() :  "";
        return provider.stamps().stream().filter(s -> s.i$(Mold.Block.Title.class)).findAny()
                                 .map(stamp -> (String) stamp.internalValue(record, provider.username()))
                                 .orElse(defaultLabel);
    }

    private static Instant group(Record record, TimeScale scale) {
        if (record == null || !record.i$(Event.class)) return null;
        return scale.normalise(record.a$(Event.class).created());
    }

    private static List<RecordItemBlock> recordItemBlockList(Record record, RecordItemBuilderProvider provider, URL baseAssetUrl) {
        return provider.blocks().stream().map(block -> recordItemBlock(record, provider, baseAssetUrl, block)).collect(toList());
    }

    private static RecordItemBlock recordItemBlock(Record record, RecordItemBuilderProvider provider, URL baseAssetUrl, Mold.Block block) {
        return new RecordItemBlock().name(block.name$()).hidden(block.hidden(record));
    }

    private static List<RecordItemStamp> recordItemStampList(Record record, RecordItemBuilderProvider provider, URL baseAssetUrl) {
        return provider.stamps().stream().map(stamp -> recordItemStamp(record, provider, baseAssetUrl, stamp)).collect(toList());
    }

    private static RecordItemStamp recordItemStamp(Record record, RecordItemBuilderProvider provider, URL baseAssetUrl, Mold.Block.Stamp stamp) {
        return new RecordItemStamp().name(stamp.name$())
                         .values(valuesOf(stamp, record, provider, baseAssetUrl))
                         .propertyList(propertiesOf(stamp, record, baseAssetUrl));
    }

    private static List<String> valuesOf(Mold.Block.Stamp stamp, Record record, RecordItemBuilderProvider provider, URL baseAssetUrl) {
        Object value = stamp.internalValue(record, provider.username());

        if (value instanceof List) {
            List<Object> values = (List<Object>) value;
            if (values.isEmpty() && stamp.i$(Picture.class)) values = singletonList(stamp.a$(Picture.class).defaultPicture());
            return values.stream().map(v -> valueOf(stamp, v, baseAssetUrl)).collect(toList());
        }

        return singletonList(valueOf(stamp, value, baseAssetUrl));
    }

    private static String valueOf(Mold.Block.Stamp stamp, Object value, URL baseAssetUrl) {

        if (stamp.i$(Breadcrumbs.class)) {
            Tree tree = (Tree) value;
            return new Gson().toJson(tree);
        }

        if (stamp.i$(RecordLinks.class)) {
            CatalogStampRecordLinks.RecordLinks links = (CatalogStampRecordLinks.RecordLinks) value;
            return new Gson().toJson(links);
        }

        if (stamp.i$(Picture.class)) {
            if (value == null)
                value = stamp.a$(Picture.class).defaultPicture();
            return value != null ? toResource(baseAssetUrl, (URL)value).toUrl().toString() : "";
        }

        if (stamp.i$(ResourceIcon.class))
            return value != null ? toResource(baseAssetUrl, (URL) value).toUrl().toString() : "";

        return value != null ? String.valueOf(value) : "";
    }

    private static List<Property> propertiesOf(Mold.Block.Stamp stamp, Record record, URL baseAssetUrl) {
        List<Property> result = new ArrayList<>();
        String style = stamp.style(record);

        if (style != null && !style.isEmpty())
            result.add(propertyOf("style", style));

        if (stamp.i$(Highlight.class))
            result.add(propertyOf("color", stamp.a$(Highlight.class).color(record)));

        if (stamp.i$(RecordLinks.class))
            result.add(propertyOf("title", stamp.a$(RecordLinks.class).title(record)));

        if (stamp.i$(CatalogLink.class))
            result.add(propertyOf("title", stamp.a$(CatalogLink.class).title(record)));

        if (stamp.i$(Location.class)) {
            URL icon = stamp.a$(Location.class).icon(record);
            if (icon != null)
                result.add(propertyOf("icon", toResource(baseAssetUrl, icon).toUrl().toString()));
        }

        if (stamp.i$(Icon.class))
            result.add(propertyOf("title", stamp.a$(Icon.class).title(record)));

        return result;
    }

    private static Property propertyOf(String name, String value) {
        return new Property().name(name).value(value);
    }

    public interface RecordItemBuilderProvider {
        List<Mold.Block> blocks();
        List<Mold.Block.Stamp> stamps();
        String username();
        TimeScale scale();
    }
}
