package io.intino.sumus.box.displays;

import io.intino.konos.server.activity.displays.Display;
import io.intino.konos.server.activity.spark.ActivityFile;
import io.intino.sumus.analytics.TimeRange;
import io.intino.sumus.box.SumusBox;
import io.intino.sumus.box.displays.ElementViewDisplay.ExecuteItemTaskEvent;
import io.intino.sumus.box.displays.ElementViewDisplay.OpenItemDialogEvent;
import io.intino.sumus.box.displays.ElementViewDisplay.OpenItemEvent;
import io.intino.sumus.box.displays.builders.MoldBuilder;
import io.intino.sumus.box.displays.builders.RecordItemBuilder;
import io.intino.sumus.box.displays.builders.RecordItemBuilder.RecordItemBuilderProvider;
import io.intino.sumus.box.displays.notifiers.RecordDisplayNotifier;
import io.intino.sumus.box.displays.providers.RecordDisplayProvider;
import io.intino.sumus.box.schemas.*;
import io.intino.sumus.graph.Catalog;
import io.intino.sumus.graph.*;
import io.intino.sumus.graph.Mold;
import io.intino.sumus.graph.Mold.Block.*;
import io.intino.sumus.graph.NameSpace;
import io.intino.sumus.graph.functions.Resource;
import io.intino.sumus.graph.functions.Tree;
import io.intino.sumus.graph.rules.TimeScale;
import io.intino.sumus.helpers.ElementHelper;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;

public class RecordDisplay extends SumusDisplay<RecordDisplayNotifier> {
    private Mold mold;
    private Element context;
    private Record record;
    private String mode;
    private RecordDisplayProvider provider;
    private List<Consumer<OpenItemEvent>> openItemListeners = new ArrayList<>();
    private List<Consumer<OpenItemDialogEvent>> openItemDialogListeners = new ArrayList<>();
    private List<Consumer<ExecuteItemTaskEvent>> executeItemTaskListeners = new ArrayList<>();
    private boolean recordDisplaysCreated = false;
    private String emptyMessage = null;

    public RecordDisplay(SumusBox box) {
        super(box);
    }

    public void mold(Mold mold) {
        this.mold = mold;
    }

    public void context(Element context) {
        this.context = context;
    }

    public void record(Record record) {
        this.record = record;
    }

    public void mode(String mode) {
        this.mode = mode;
    }

    public void provider(RecordDisplayProvider provider) {
        this.provider = provider;
    }

    public void onOpenItem(Consumer<OpenItemEvent> listener) {
        openItemListeners.add(listener);
    }

    public void onOpenItemDialog(Consumer<OpenItemDialogEvent> parameters) {
        openItemDialogListeners.add(parameters);
    }

    public void onExecuteItemTask(Consumer<ExecuteItemTaskEvent> parameters) {
        executeItemTaskListeners.add(parameters);
    }

    @Override
    protected void init() {
        super.init();
        sendEmptyMessage();
        sendMode();
    }

    @Override
    public void refresh() {
        super.refresh();
        children(StampDisplay.class).forEach(display -> {
            display.record(record);
            if (display instanceof TemporalStampDisplay) {
                TemporalStampDisplay temporalDisplay = (TemporalStampDisplay) display;
                temporalDisplay.nameSpace(provider.nameSpace());
                temporalDisplay.range(provider.range());
            }
            display.refresh();
        });
        remove(PageContainerDisplay.class);
        sendInfo();
    }

    public void refresh(RecordItem item) {
        sendInfo(item);
    }

    public void itemStampsReady(String id) {
        if (!recordDisplaysCreated)
            displays(record).forEach((key, display) -> {
                add(display);
                display.record(record);
                display.personifyOnce(id + key.name$());
                display.refresh();
            });
        pages(record).forEach(display -> {
            add(display);
            display.personifyOnce(id);
            display.refresh();
        });
        embeddedCatalogs().forEach((key, display) -> {
            display.filter(record -> key.filter(context, RecordDisplay.this.record, (Record) record));
            display.nameSpaceHandler(provider.nameSpaceHandler());
            display.label(key.label());
            display.onOpenItem(params -> notifyOpenItem((OpenItemEvent) params));
            display.embedded(true);
            add(display);
            display.personifyOnce(id + key.name$());
        });
        recordDisplaysCreated = true;
    }

    public void notifyOpenItem(OpenItemEvent params) {
        openItemListeners.forEach(l -> l.accept(params));
    }

    public void selectElement(Item stampItem) {
        io.intino.sumus.graph.Mold.Block.Stamp stamp = provider.stamp(mold, stampItem.name());
        if (!stamp.i$(CatalogLink.class)) return;

        CatalogLink catalogLinkStamp = stamp.a$(CatalogLink.class);
        ElementDisplay display = provider.openElement(catalogLinkStamp.catalog().label());

        display.filterAndNotify(record -> catalogLinkStamp.filter(this.record, (Record)record));
        if (display instanceof TemporalCatalogDisplay)
            ((TemporalCatalogDisplay) display).selectNameSpaceAndRange(provider.nameSpace(), provider.range());

        display.refresh();
    }

    public void openItemDialogOperation(OpenItemDialogParameters params) {
        openItemDialogListeners.forEach(l -> l.accept(ElementHelper.openItemDialogEvent(params.item(), provider.stamp(mold, params.stamp()))));
    }

    public void executeItemTaskOperation(ExecuteItemTaskParameters params) {
        executeItemTaskListeners.forEach(l -> l.accept(ElementHelper.executeItemTaskEvent(params.item(), provider.stamp(mold, params.stamp()))));
    }

    public io.intino.konos.server.activity.spark.ActivityFile downloadItemOperation(DownloadItemParameters parameters) {
        io.intino.sumus.graph.Mold.Block.Stamp stamp = provider.stamps(mold).stream().filter(s -> s.name$().equals(parameters.stamp())).findFirst().orElse(null);
        Resource resource = stamp.a$(DownloadOperation.class).execute(record, parameters.option());
        return new ActivityFile() {
            @Override
            public String label() {
                return resource.label();
            }

            @Override
            public InputStream content() {
                return resource.content();
            }
        };
    }

    public ActivityFile exportItemOperation(ExportItemParameters parameters) {
        io.intino.sumus.graph.Mold.Block.Stamp stamp = provider.stamps(mold).stream().filter(s -> s.name$().equals(parameters.stamp())).findFirst().orElse(null);
        Resource resource = stamp.a$(ExportOperation.class).execute(record, parameters.from(), parameters.to(), parameters.option(), username());
        return new ActivityFile() {
            @Override
            public String label() {
                return resource.label();
            }

            @Override
            public InputStream content() {
                return resource.content();
            }
        };
    }

    public void executeOperation(ElementOperationParameters value) {
        provider.executeOperation(value, singletonList(record));
    }

    public io.intino.konos.server.activity.spark.ActivityFile downloadOperation(ElementOperationParameters value) {
        Resource resource = provider.downloadOperation(value, singletonList(record));
        return new ActivityFile() {
            @Override
            public String label() {
                return resource.label();
            }

            @Override
            public InputStream content() {
                return resource.content();
            }
        };
    }

    private void sendInfo(RecordItem item) {
        notifier.refresh(new RecordRefreshInfo().mold(MoldBuilder.build(mold)).recordItem(item));
    }

    private void sendInfo() {
        sendInfo(RecordItemBuilder.build(record, new RecordItemBuilderProvider() {
            @Override
            public List<Mold.Block> blocks() {
                return provider.blocks(mold);
            }

            @Override
            public List<Mold.Block.Stamp> stamps() {
                return provider.stamps(mold);
            }

            @Override
            public String username() {
                return RecordDisplay.this.username();
            }

            @Override
            public TimeScale scale() {
                return provider.range() != null ? provider.range().scale() : null;
            }
        }, baseAssetUrl()));
    }

    private void sendEmptyMessage() {
        if (emptyMessage == null) return;
        notifier.refreshEmptyMessage(emptyMessage);
    }

    private void sendMode() {
        notifier.refreshMode(mode);
    }

    private Map<Mold.Block.Display, StampDisplay> displays(Record record) {
        List<Mold.Block.Stamp> stamps = provider.stamps(mold).stream().filter(s -> s.i$(Mold.Block.Display.class)).collect(toList());
        Map<Mold.Block.Display, StampDisplay> mapWithNulls = stamps.stream().collect(HashMap::new, (map, stamp)->map.put(stamp.a$(Mold.Block.Display.class), provider.display(stamp.name$())), HashMap::putAll);
        Map<Mold.Block.Display, StampDisplay> result = mapWithNulls.entrySet().stream().filter(e -> e.getValue() != null).collect(toMap(Map.Entry::getKey, Map.Entry::getValue));
        result.forEach((key, value) -> value.record(record));
        return result;
    }

    private List<Display> pages(Record record) {
        return provider.stamps(mold).stream().filter(s -> s.i$(Page.class))
                .map(stamp -> new PageContainerDisplay(box).pageLocation(location(stamp, record)))
                .collect(Collectors.toList());
    }

    private Map<EmbeddedCatalog, CatalogDisplay> embeddedCatalogs() {
        List<Mold.Block.Stamp> stamps = provider.stamps(mold).stream().filter(s -> s.i$(EmbeddedCatalog.class)).collect(toList());
        return stamps.stream().collect(Collectors.toMap(s -> s.a$(EmbeddedCatalog.class), s -> displayFor(s.a$(EmbeddedCatalog.class).catalog())));
    }

    private PageLocation location(io.intino.sumus.graph.Mold.Block.Stamp stamp, Record record) {
        PageLocation location = new PageLocation();
        if (stamp.i$(ExternalPage.class)) return location.value(stamp.a$(ExternalPage.class).url(record.name$()).toString()).internal(false);
        return location.value(stamp.a$(InternalPage.class).path(record.name$())).internal(true);
    }

    private CatalogDisplay displayFor(Catalog catalog) {
        CatalogDisplay result = null;

        if (catalog.isEntityHolder()) result = new EntityCatalogDisplay(box);
        else if (catalog.isReportHolder()) result = new ReportCatalogDisplay(box);
        else if (catalog.isEventHolder()) result = new EventCatalogDisplay(box);

        if (result == null)
            return null;

        result.target(record);
        result.catalog(catalog);

        return result;
    }

    public void selectRecord(Item item) {
        openItemListeners.forEach(l -> l.accept(new OpenItemEvent() {
            @Override
            public String item() {
                return item.name();
            }

            @Override
            public String label() {
                return item.label();
            }

            @Override
            public Record record() {
                return null;
            }

            @Override
            public Panel panel() {
                return (Panel) context;
            }

            @Override
            public NameSpace nameSpace() {
                return provider.nameSpace();
            }

            @Override
            public TimeRange range() {
                return provider.range();
            }

            @Override
            public Tree breadcrumbs() {
                return null;
            }
        }));
    }

    public void saveItem(SaveItemParameters value) {
        provider.saveItem(value, record);
    }

    public void emptyMessage(String message) {
        this.emptyMessage = message;
    }
}