package io.intino.sumus.box.displays;

import io.intino.konos.server.activity.displays.DisplayNotifier;
import io.intino.konos.server.activity.services.push.User;
import io.intino.sumus.Settings;
import io.intino.sumus.box.SumusBox;
import io.intino.sumus.box.displays.builders.ItemBuilder;
import io.intino.sumus.box.displays.builders.PlatformInfoBuilder;
import io.intino.sumus.box.schemas.ItemProperty;
import io.intino.sumus.box.schemas.PlatformInfo;
import io.intino.sumus.box.schemas.UserInfo;
import io.intino.sumus.graph.*;
import io.intino.sumus.graph.entityholder.EntityHolderCatalog;
import io.intino.sumus.graph.eventholder.EventHolderCatalog;
import io.intino.sumus.graph.reportholder.ReportHolderCatalog;
import io.intino.tara.magritte.Node;

import java.util.*;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Stream;

import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toList;

public abstract class LayoutDisplay<DN extends DisplayNotifier> extends ElementRepositoryDisplay<DN> {
	private List<Item> items = new ArrayList<>();
	private List<Consumer<Boolean>> loadingListeners = new ArrayList<>();
	private List<Consumer<Boolean>> loadedListeners = new ArrayList<>();
	protected Map<Class<? extends ElementRender>, Function<ElementRender, List<Item>>> itemsProviders = new HashMap<>();

	public LayoutDisplay(SumusBox box) {
		super(box);
		buildElementProviders();
	}

	public void onLoading(Consumer<Boolean> listener) {
		loadingListeners.add(listener);
	}

	public void onLoaded(Consumer<Boolean> listener) {
		loadedListeners.add(listener);
	}

	@Override
	protected void refreshLoading(boolean withMessage) {
		sendLoading();
		loadingListeners.forEach(l -> l.accept(withMessage));
	}

	@Override
	protected void refreshLoaded() {
		sendLoaded();
		loadedListeners.forEach(l -> l.accept(true));
	}

	public void logout() {
		session().logout();
		notifyLoggedOut();
	}

	@Override
	protected void init() {
		super.init();
		sendInfo(info());
		user().ifPresent(user -> sendUser(userOf(user)));
		sendItems();
		sendNameSpaceList();
		notifyLoaded();
	}

	@Override
	protected Element element(String label) {
		Item item = item(label);
		return item != null ? item.element() : null;
	}

	@Override
	protected Record target(String label) {
		Item item = item(label);
		return item != null ? item.target() : null;
	}

	protected Item item(String label) {
		return items.stream().filter(i -> i.label().equals(label)).findFirst().orElse(null);
	}

	public abstract void selectItem(String label);

	protected abstract void sendLoading();
	protected abstract void sendLoaded();
	protected abstract void sendUser(UserInfo userInfo);
	protected abstract void sendInfo(PlatformInfo info);
	protected abstract void sendItems(List<Item> itemList);
	protected abstract void notifyLoggedOut();
	protected abstract void refreshNameSpaces(List<NameSpace> nameSpaces);
	protected abstract void refreshSelectedNameSpace(NameSpace nameSpace);

	protected io.intino.sumus.box.schemas.Item schemaItemOf(Item item) {
		io.intino.sumus.box.schemas.Item result = ItemBuilder.build(item.name(), item.label());
		List<ItemProperty> itemProperties = result.itemPropertyList();
		itemProperties.add(new ItemProperty().name("type").value(typeOf(item.element())));
		itemProperties.add(new ItemProperty().name("group").value(item.group() != null ? item.group().label() : null));
		itemProperties.add(new ItemProperty().name("group_opened").value(item.group() != null ? String.valueOf(item.group().mode() == Group.Mode.Expanded) : null));
		return result;
	}

	private List<Item> itemOf(ElementOption option) {
		ElementRender render = renderOf(option);
		if (render == null) return emptyList();
		return itemsProviders.get(render.getClass()).apply(render);
	}

	private ElementRender renderOf(ElementOption option) {
		if (option.i$(Option.class)) return option.a$(Option.class).elementRender();
		else if (option.i$(Options.class)) return option.a$(Options.class).elementRender();
		return null;
	}

	private String labelOf(ElementRender render, Element element, Record record) {
		Node owner = render.core$().owner();

		if (owner.is(Option.class)) return owner.as(Option.class).label();
		else if (owner.is(Group.class)) return owner.as(Option.class).label();
		else if (owner.is(Options.class)) return owner.as(Options.class).label(element, record);

		return "no label";
	}

	private List<Item> items() {
		return optionList().map(this::itemOf).flatMap(Collection::stream).collect(toList());
	}

	private Stream<ElementOption> optionList() {
		return platform().layout().elementOptionList().stream().map(v -> {
			if (v.i$(Group.class)) return v.a$(Group.class).elementOptionList();
			return singletonList(v);
		}).flatMap(Collection::stream);
	}

	private static String typeOf(Element element) {
		if (element == null) return "";
		if (element.i$(Panel.class)) return "panel";
		if (element.i$(EntityHolderCatalog.class)) return "entity";
		if (element.i$(EventHolderCatalog.class)) return "event";
		if (element.i$(ReportHolderCatalog.class)) return "report";
		return "";
	}

	private void sendItems() {
		this.items = items();
		sendItems(items);
	}

	private List<Item> panelItem(ElementRender r) {
		Panel panel = r.a$(RenderPanel.class).panel();
		return singletonList(itemOf(r, panel));
	}

	private List<Item> panelItems(ElementRender r) {
		List<Panel> panelList = r.a$(RenderPanels.class).source();
		return panelList.stream().map(c -> itemOf(r, c)).collect(toList());
	}

	private List<Item> recordItem(ElementRender r) {
		RenderRecord recordViewCollection = r.a$(RenderRecord.class);
		Record record = recordViewCollection.record();
		return singletonList(itemOf(r, recordViewCollection.container(), record));
	}

	private List<Item> recordItems(ElementRender r) {
		RenderRecords recordViewCollection = r.a$(RenderRecords.class);
		List<Record> recordList = recordViewCollection.source();
		return recordList.stream().map(record -> itemOf(r, recordViewCollection.panel(), record)).collect(toList());
	}

	private List<Item> catalogItem(ElementRender r) {
		Catalog catalog = r.a$(RenderCatalog.class).catalog();
		return singletonList(itemOf(r, catalog));
	}

	private List<Item> catalogItems(ElementRender r) {
		RenderCatalogs render = r.a$(RenderCatalogs.class);
		return platform().catalogList().stream().filter(render::filter).map(c -> itemOf(r, c)).collect(toList());
	}

	private List<Item> olapItems(ElementRender r) {
		RenderOlap render = r.a$(RenderOlap.class);
		return singletonList(itemOf(r, render.olap()));
	}

	private List<Item> moldItems(ElementRender r) {
		RenderMold render = r.a$(RenderMold.class);
		return singletonList(itemOf(r, render.mold()));
	}

	private PlatformInfo info() {
		Settings settings = box.actionsConfiguration();
		return PlatformInfoBuilder.build(settings, box.configuration().platformConfiguration().authServiceUrl());
	}

	private UserInfo userOf(User user) {
		return new UserInfo().fullName(user.fullName()).photo(user.photo().toString());
	}

	private void notifyLoaded() {
		loadedListeners.forEach(l -> l.accept(true));
	}

	private void buildElementProviders() {
		itemsProviders.put(RenderPanel.class, this::panelItem);
		itemsProviders.put(RenderPanels.class, this::panelItems);
		itemsProviders.put(RenderRecord.class, this::recordItem);
		itemsProviders.put(RenderRecords.class, this::recordItems);
		itemsProviders.put(RenderCatalog.class, this::catalogItem);
		itemsProviders.put(RenderCatalogs.class, this::catalogItems);
		itemsProviders.put(RenderOlap.class, this::olapItems);
		itemsProviders.put(RenderMold.class, this::moldItems);
	}

	private Item itemOf(ElementRender render, Element element) {
		return itemOf(render, element, null);
	}

	private Item itemOf(ElementRender render, Element element, Record target) {
		return new Item() {
			@Override
			public String name() {
				return target != null ? target.name$() : element.name$();
			}

			@Override
			public String label() {
				return labelOf(render, element, target);
			}

			@Override
			public Group group() {
				return render.core$().ownerAs(Group.class);
			}

			@Override
			public Element element() {
				return element;
			}

			@Override
			public Record target() {
				return target;
			}
		};
	}

	private void sendNameSpaceList() {
		refreshNameSpaces(nameSpaceHandler().nameSpaces(username()));
		refreshSelectedNameSpace(nameSpaceHandler().selectedNameSpace(username()));
	}

	protected interface Item {
		String name();
		String label();
		Group group();
		Element element();
		Record target();
	}

}