package io.intino.sumus.box.displays;

import com.google.gson.Gson;
import io.intino.konos.server.activity.displays.DisplayNotifier;
import io.intino.sumus.RecordList;
import io.intino.sumus.analytics.TimeRange;
import io.intino.sumus.box.SumusBox;
import io.intino.sumus.box.displays.ElementViewDisplay.ExecuteItemTaskEvent;
import io.intino.sumus.box.displays.ElementViewDisplay.OpenItemDialogEvent;
import io.intino.sumus.box.displays.ElementViewDisplay.OpenItemEvent;
import io.intino.sumus.box.displays.builders.RecordItemBuilder.RecordItemBuilderProvider;
import io.intino.sumus.box.schemas.ElementOperationParameters;
import io.intino.sumus.box.schemas.RecordItem;
import io.intino.sumus.box.schemas.SaveItemParameters;
import io.intino.sumus.graph.*;
import io.intino.sumus.graph.Catalog.MoldView;
import io.intino.sumus.graph.Toolbar.*;
import io.intino.sumus.graph.functions.Resource;
import io.intino.sumus.graph.functions.StampSaveEvent;
import io.intino.sumus.graph.functions.Tree;
import io.intino.sumus.graph.rules.TimeScale;
import io.intino.sumus.helpers.ElementHelper;
import io.intino.sumus.helpers.NameSpaceHandler;

import java.util.*;
import java.util.function.Consumer;
import java.util.function.Function;

import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toList;

public abstract class ElementDisplay<E extends Element, DN extends DisplayNotifier> extends SumusDisplay<DN> implements RecordItemBuilderProvider {
	private String label;
	private E element;
	private Record target;
	private NameSpaceHandler nameSpaceHandler = null;
	private ElementDisplayManager elementDisplayManager = null;
	private List<Consumer<Boolean>> loadingListeners = new ArrayList<>();
	private ElementViewDisplay currentView = null;
	private Function<Record, Boolean> recordFilter = null;
	private Boolean dirty = null;
	private boolean embedded = false;
	private OpenItemEvent openedItem = null;
	private TimeRange range;

	public ElementDisplay(SumusBox box) {
		super(box);
	}

	public String label() {
		return this.label;
	}

	public void label(String label) {
		this.label = label;
	}

	public E element() {
		return this.element;
	}

	public void element(E element) {
		this.element = element;
	}

	public Record target() {
		return this.target;
	}

	public void target(Record target) {
		this.target = target;
	}

	public ElementDisplayManager elementDisplayManager() {
		return this.elementDisplayManager;
	}

	public void elementDisplayManager(ElementDisplayManager manager) {
		this.elementDisplayManager = manager;
	}

	public NameSpaceHandler nameSpaceHandler() {
		if (this.nameSpaceHandler == null) this.nameSpaceHandler = new NameSpaceHandler(box.graph());
		return this.nameSpaceHandler;
	}

	public NameSpace nameSpace() {
		return nameSpaceHandler().selectedNameSpace(username());
	}

	public TimeRange range() {
		return range;
	}

	public void range(TimeRange range) {
		this.range = range;
	}

	public void nameSpaceHandler(NameSpaceHandler nameSpaceHandler) {
		this.nameSpaceHandler = nameSpaceHandler;
		this.nameSpaceHandler.onSelect(this::selectNameSpace);
	}

	public boolean embedded() {
		return this.embedded;
	}

	public void embedded(boolean value) {
		this.embedded = value;
	}

	public void onLoading(Consumer<Boolean> listener) {
		loadingListeners.add(listener);
	}

	public void filter(Function<Record, Boolean> recordFilter) {
		this.recordFilter = recordFilter;
		dirty(true);
	}

	public void filterAndNotify(Function<Record, Boolean> recordFilter) {
		filter(recordFilter);
		notifyFiltered(recordFilter != null);
	}

	public void clearFilter() {
		boolean refresh = recordFilter != null;
		filterAndNotify(null);
		if (refresh) refresh();
	}

	public List<Mold.Block> blocks() {
		return molds().stream().map(this::blocks).flatMap(Collection::stream).collect(toList());
	}

	public List<Mold.Block> blocks(Mold mold) {
		return mold.blockList().stream().map(this::blocks).flatMap(Collection::stream).collect(toList());
	}

	public List<Mold.Block> blocks(Mold.Block block) {
		List<Mold.Block> result = new ArrayList<>();
		result.add(block);
		block.blockList().forEach(child -> result.addAll(blocks(child)));
		return result;
	}

	public List<Mold.Block.Stamp> stamps() {
		return molds().stream().map(Mold::blockList).flatMap(Collection::stream).map(this::stamps).flatMap(Collection::stream).collect(toList());
	}

	public List<Mold.Block.Stamp> stamps(Mold mold) {
		return mold.blockList().stream().map(this::stamps).flatMap(Collection::stream).collect(toList());
	}

	public List<Mold.Block.Stamp> stamps(Mold.Block block) {
		List<Mold.Block.Stamp> stamps = new ArrayList<>();

		stamps.addAll(block.stampList());
		block.blockList().forEach(child -> stamps.addAll(stamps(child)));

		return stamps;
	}

	public List<Mold.Block.Stamp> expandedStamps() {
		return molds().stream().map(Mold::blockList).flatMap(Collection::stream).filter(Mold.Block::isExpanded).map(this::stamps).flatMap(Collection::stream).collect(toList());
	}

	public List<Mold.Block.Stamp> expandedStamps(Mold mold) {
		return mold.blockList().stream().filter(Mold.Block::isExpanded).map(this::stamps).flatMap(Collection::stream).collect(toList());
	}

	public Mold.Block.Stamp stamp(Mold mold, String name) {
		return stamps(mold).stream().filter(s -> s.name$().equals(name)).findFirst().orElse(null);
	}

	public Mold.Block.Stamp stamp(String name) {
		return stamps().stream().filter(s -> s.name$().equals(name)).findFirst().orElse(null);
	}

	public StampDisplay display(String stamp) {
		return box.stampDisplayProvider() != null ? box.stampDisplayProvider().display(stamp) : null;
	}

	public void executeOperation(ElementOperationParameters params, List<Record> selection) {
		Toolbar.Operation operation = operationOf(params);
		executeOperation(operation, params.option(), selection);
	}

	public Resource downloadOperation(ElementOperationParameters params, List<Record> selection) {
		Toolbar.Operation operation = operationOf(params);
		return downloadOperation(operation, params, selection);
	}

	public void saveItem(SaveItemParameters params, Record record) {
		Mold.Block.Stamp stamp = stamp(params.stamp());
		if (!stamp.isEditable()) return;

		currentRecord(new String(Base64.getDecoder().decode(params.item())));
		StampSaveEvent.Refresh refresh = stamp.asEditable().onSave(record, params.value(), username());
		currentView().ifPresent(view -> {
			dirty(true);
			if (refresh == StampSaveEvent.Refresh.Record) view.refresh(currentItem());
			else if (refresh == StampSaveEvent.Refresh.Catalog) view.refresh();
		});
	}

	public Optional<ElementViewDisplay> currentView() {
		return Optional.ofNullable(currentView);
	}

	public TimeScale scale() {
		return null;
	}

	@Override
	public void refresh() {
		refreshView();
	}

	public void refreshView() {
		currentView().ifPresent(ElementViewDisplay::refresh);
	}

	public void refresh(Record... records) {
		currentView().ifPresent(v -> v.refresh(ElementHelper.recordItems(records, this, baseAssetUrl())));
	}

	public boolean dirty() {
		return dirty == null || dirty;
	}

	public void dirty(boolean value) {
		dirty = value;
	}

	public void navigate(String key) {
		if (!key.equals("main")) return;
		hidePanel();
		refreshBreadcrumbs("");
	}

	public void selectInstant(CatalogInstantBlock block) {
		CatalogDisplay display = catalogDisplayOf(block);
		List<String> entities = block.entities();
		display.filterAndNotify(record -> entities.contains(((Record)record).core$().id()));
		display.refreshViews();
	}

	private CatalogDisplay catalogDisplayOf(CatalogInstantBlock block) {
		if (!this.element.name$().equals(block.catalog()) && !this.element.label().equals(block.catalog()))
			return openElement(block.catalog());

		CatalogDisplay display = (CatalogDisplay) this;
		List<?> views = display.views();
		Catalog.View view = views.stream().map(v -> (Catalog.View)v).filter(v -> !v.i$(Catalog.OlapView.class)).findFirst().orElse(null);
		if (view != null) display.selectView(view.name$());

		return display;
	}

	protected void notifyLoading(Boolean loading) {
		loadingListeners.forEach(l -> l.accept(loading));
	}

	protected List<? extends AbstractView> views() {
		E element = element();

		if (element.i$(Catalog.class)) {
			Catalog catalog = element.a$(Catalog.class);
			return catalog.view() != null ? singletonList(catalog.view()) : catalog.views().viewList();
		}
		else if (element.i$(Panel.class)) {
			Panel panel = element.a$(Panel.class);
			return panel.view() != null ? singletonList(panel.view()) : panel.views().viewList();
		}

		return Collections.emptyList();
	}

	protected List<Mold> molds() {
		return views().stream().filter(v -> v.i$(MoldView.class)).map(v -> v.a$(MoldView.class).mold()).collect(toList());
	}

	protected void updateCurrentView(ElementViewDisplay display) {
		this.currentView = display;
		refreshView();
	}

	protected void createDialogContainer() {
		DialogContainerDisplay display = new DialogContainerDisplay(box);
		display.onDialogAssertion((modification) -> currentView().ifPresent(view -> {
			dirty(true);
			if (modification.toLowerCase().equals("itemmodified")) view.refresh(currentItem());
			else if (modification.toLowerCase().equals("catalogmodified")) view.refresh();
		}));
		add(display);
		display.personifyOnce();
	}

	protected void openItem(OpenItemEvent event) {
		openedItem = event;
		showPanel();
		createPanel(event.item());
		PanelDisplay display = createPanelDisplay(event);
		add(display);
		display.personifyOnce(event.item());
		refreshBreadcrumbs(breadcrumbs(event));
	}

	protected void openItemDialog(OpenItemDialogEvent event) {
		currentRecord(new String(Base64.getDecoder().decode(event.item())));

		DialogContainerDisplay display = child(DialogContainerDisplay.class);
		display.dialogWidth(event.width());
		display.dialogHeight(event.height());
		display.dialogLocation(event.path());
		display.refresh();
		showDialog();
	}

	protected void executeItemTask(ExecuteItemTaskEvent event) {
		currentRecord(new String(Base64.getDecoder().decode(event.item())));
		Record record = currentRecord();
		event.stamp().a$(Mold.Block.TaskOperation.class).task(record, username());
		dirty(true);
		refresh(currentRecord());
	}

	public <E extends ElementDisplay> E openElement(String label) {
		return elementDisplayManager.openElement(label);
	}

	public PanelDisplay createPanelDisplay(OpenItemEvent event) {
		PanelDisplay display = elementDisplayManager.createElement(event.panel(), event.record());
		display.range(event.range());
		return display;
	}

	public Record record(String key) {
		return loadRecord(key);
	}

	protected abstract void showDialog();
	protected abstract void currentRecord(String id);
	protected abstract Record currentRecord();
	protected abstract void notifyFiltered(boolean value);
	protected abstract void selectNameSpace(NameSpace nameSpace);
	protected abstract Record loadRecord(String id);
	protected abstract void refreshBreadcrumbs(String breadcrumbs);
	protected abstract void createPanel(String item);
	protected abstract void showPanel();
	protected abstract void hidePanel();

	protected void applyFilter(RecordList recordList) {
		if (recordFilter == null) return;
		recordList.filter(recordFilter);
	}

	private RecordItem currentItem() {
		return ElementHelper.recordItem(currentRecord(), this, baseAssetUrl());
	}

	private Toolbar.Operation operationOf(ElementOperationParameters params) {
		Optional<Toolbar> toolbar = toolbar();

		if (!toolbar.isPresent())
			return null;

		return toolbar.get().operationList().stream().filter(op -> op.name$().equals(params.operation())).findFirst().orElse(null);
	}

	private Optional<Toolbar> toolbar() {
		E element = element();

		if (element.i$(Catalog.class)) return Optional.ofNullable(element.a$(Catalog.class).toolbar());
		if (element.i$(Panel.class)) return Optional.ofNullable(element.a$(Panel.class).toolbar());

		return Optional.empty();
	}

	private void executeOperation(Toolbar.Operation operation, String option, List<Record> selection) {
		if (operation.i$(OpenDialog.class)) {
			DialogContainerDisplay display = child(DialogContainerDisplay.class);
			OpenDialog openDialog = operation.a$(OpenDialog.class);
			display.dialogWidth(openDialog.width());
			display.dialogHeight(openDialog.height());
			display.dialogLocation(openDialog.path());
			display.refresh();
			showDialog();
		}

		if (operation.i$(Task.class)) {
			Task taskOperation = operation.a$(Task.class);
			taskOperation.execute(element().a$(Catalog.class), option, username());
			if (taskOperation.refresh() == Task.Refresh.Catalog)
				this.refresh();
			return;
		}

		if (operation.i$(TaskSelection.class)) {
			TaskSelection taskSelectionOperation = operation.a$(TaskSelection.class);
			taskSelectionOperation.execute(element().a$(Catalog.class), option, selection, username());
			if (taskSelectionOperation.refresh() == TaskSelection.Refresh.Catalog)
				this.refresh();
			else if (taskSelectionOperation.refresh() == TaskSelection.Refresh.Selection)
				this.refresh(selection.toArray(new Record[selection.size()]));
		}
	}

	private Resource downloadOperation(Toolbar.Operation operation, ElementOperationParameters params, List<Record> selection) {
		E element = element();

		if (operation.i$(Export.class))
			return operation.a$(Export.class).execute(element, params.from(), params.to());

		if (operation.i$(ExportSelection.class))
			return operation.a$(ExportSelection.class).execute(element, params.from(), params.to(), selection);

		if (operation.i$(Download.class))
			return operation.a$(Download.class).execute(element, params.option());

		if (operation.i$(DownloadSelection.class))
			return operation.a$(DownloadSelection.class).execute(element, params.option(), selection);

		return null;
	}

	private String breadcrumbs(OpenItemEvent event) {
		Tree tree = event.breadcrumbs();

		if (tree == null) {
			tree = new Tree();
			Tree.TreeItem main = new Tree.TreeItem().name("main").label(label());
			if (openedItem != null)
				main.add(new Tree.TreeItem().name(openedItem.record().name$()).label(openedItem.label()));
			tree.add(main);
		}

		return new Gson().toJson(tree);
	}

}
