package io.intino.sumus.box.displays;

import io.intino.konos.server.activity.spark.ActivityFile;
import io.intino.sumus.analytics.viewmodels.ElementView;
import io.intino.sumus.box.SumusBox;
import io.intino.sumus.box.displays.notifiers.CatalogMagazineViewDisplayNotifier;
import io.intino.sumus.box.displays.providers.CatalogViewDisplayProvider;
import io.intino.sumus.box.schemas.ElementOperationParameters;
import io.intino.sumus.box.schemas.RecordItem;
import io.intino.sumus.graph.Cluster;
import io.intino.sumus.graph.Record;
import io.intino.sumus.graph.functions.Resource;
import io.intino.tara.magritte.Predicate;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.function.Consumer;

import static io.intino.sumus.helpers.ElementHelper.recordDisplayProvider;
import static java.util.Collections.singletonList;

public class CatalogMagazineViewDisplay extends SumusDisplay<CatalogMagazineViewDisplayNotifier> implements CatalogViewDisplay {
    private ElementView view;
    private CatalogViewDisplayProvider provider;
    private List<Consumer<OpenItemDialogEvent>> openItemDialogListeners = new ArrayList<>();
    private List<Consumer<ExecuteItemTaskEvent>> executeItemTaskListeners = new ArrayList<>();
    private Record record = null;
    private String condition = null;
    private List<Consumer<Boolean>> loadingListeners = new ArrayList<>();
    private String currentRecord = null;

    public CatalogMagazineViewDisplay(SumusBox box) {
        super(box);
    }

    @Override
    public void view(ElementView view) {
        this.view = view;
    }

    @Override
    public void provider(CatalogViewDisplayProvider provider) {
        this.provider = provider;
    }

    @Override
    public void onOpenItemDialog(Consumer<OpenItemDialogEvent> parameters) {
        openItemDialogListeners.add(parameters);
    }

    @Override
    public void onExecuteItemTask(Consumer<ExecuteItemTaskEvent> parameters) {
        executeItemTaskListeners.add(parameters);
    }

    @Override
    public void onCreateCluster(Consumer<Cluster> listener) {
    }

    @Override
    public void onOpenItem(Consumer<OpenItemEvent> listener) {
    }

    @Override
    public void reset() {
        currentRecord = null;
    }

    @Override
    public void onLoading(Consumer<Boolean> listener) {
        this.loadingListeners.add(listener);
    }

    @Override
    public ElementView view() {
        return view;
    }

    @Override
    public void refresh(RecordItem... items) {
        if (items.length <= 0) return;
        child(RecordDisplay.class).refresh(items[0]);
    }

    @Override
    protected void init() {
        super.init();
        createRecordDisplay();
    }

    private void createRecordDisplay() {
        RecordDisplay display = new RecordDisplay(box);
        display.emptyMessage(view.emptyMessage());
        display.mold(view.mold());
        display.context(provider.element());
        display.record(null);
        display.mode("magazine");
        display.provider(recordDisplayProvider(provider, view));
        display.onOpenItem(this::selectRecord);
        display.onOpenItemDialog(this::openItemDialogOperation);
        display.onExecuteItemTask(this::executeItemTaskOperation);
        add(display);
        display.personifyOnce();
    }

    @Override
    public void refresh() {
        super.refresh();
        notifyLoading(true);
        loadRecord();
        if (record == null) return;
        RecordDisplay recordDisplay = child(RecordDisplay.class);
        recordDisplay.record(record);
        recordDisplay.refresh();
        notifyLoading(false);
    }

    public void filter(String value) {
        this.condition = value;
        this.loadRecord();
        this.refresh();
    }

    public void selectRecord(OpenItemEvent params) {
        currentRecord = params.item();
        refresh();
    }

    private void loadRecord() {
        int count = provider.countRecords(condition);
        record = currentRecord != null ? provider.record(nameOf(currentRecord)) : provider.rootRecord(provider.records(0, count, condition));
        currentRecord = record != null ? record.core$().id() : currentRecord;
        record = record != null ? record : provider.defaultRecord(nameOf(currentRecord));
    }

    private String nameOf(String currentRecord) {
        if (currentRecord == null) return null;
        return Predicate.nameOf(currentRecord);
    }

    private void notifyLoading(boolean value) {
        loadingListeners.forEach(l -> l.accept(value));
    }

    public void openItemDialogOperation(OpenItemDialogEvent event) {
        openItemDialogListeners.forEach(l -> l.accept(event));
    }

    public void executeItemTaskOperation(ExecuteItemTaskEvent event) {
        executeItemTaskListeners.forEach(l -> l.accept(event));
    }

    public void executeOperation(ElementOperationParameters value) {
        provider.executeOperation(value, singletonList(record));
    }

    public io.intino.konos.server.activity.spark.ActivityFile downloadOperation(ElementOperationParameters value) {
        Resource resource = provider.downloadOperation(value, singletonList(record));
        return new ActivityFile() {
            @Override
            public String label() {
                return resource.label();
            }

            @Override
            public InputStream content() {
                return resource.content();
            }
        };
    }

}