package io.intino.sumus.box.displays;

import io.intino.konos.server.activity.displays.Display;
import io.intino.konos.server.activity.displays.PageDisplay;
import io.intino.konos.server.activity.services.push.User;
import io.intino.konos.server.activity.spark.ActivityFile;
import io.intino.sumus.analytics.TimeRange;
import io.intino.sumus.analytics.viewmodels.ElementView;
import io.intino.sumus.box.SumusBox;
import io.intino.sumus.box.displays.builders.CatalogSortingBuilder;
import io.intino.sumus.box.displays.builders.ElementViewBuilder;
import io.intino.sumus.box.displays.builders.PictureDataBuilder;
import io.intino.sumus.box.displays.builders.RecordItemBuilder;
import io.intino.sumus.box.displays.notifiers.CatalogListViewDisplayNotifier;
import io.intino.sumus.box.displays.providers.CatalogViewDisplayProvider;
import io.intino.sumus.box.schemas.*;
import io.intino.sumus.graph.Catalog;
import io.intino.sumus.graph.*;
import io.intino.sumus.graph.Mold;
import io.intino.sumus.graph.Mold.Block.Picture;
import io.intino.sumus.graph.Mold.Block.Title;
import io.intino.sumus.graph.NameSpace;
import io.intino.sumus.graph.functions.Resource;
import io.intino.sumus.graph.functions.Tree;
import spark.utils.IOUtils;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.*;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static io.intino.sumus.helpers.Asset.baseAssetUrl;
import static io.intino.sumus.helpers.ClusterHelper.registerClusterGroup;
import static io.intino.sumus.helpers.ElementHelper.*;
import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;

public class CatalogListViewDisplay extends PageDisplay<CatalogListViewDisplayNotifier> implements CatalogViewDisplay {
    private SumusBox box;
    private CatalogViewDisplayProvider.Sorting sorting;
    private ElementView<Catalog> view;
    private CatalogViewDisplayProvider provider;
    private List<Consumer<OpenItemEvent>> openItemListeners = new ArrayList<>();
    private List<Consumer<OpenItemDialogEvent>> openItemDialogListeners = new ArrayList<>();
    private List<Consumer<ExecuteItemTaskEvent>> executeItemTaskListeners = new ArrayList<>();
    private List<Consumer<io.intino.sumus.graph.Cluster>> createClusterListeners = new ArrayList<>();
    private String condition = null;
    private Map<String, List<StampDisplay>> recordDisplaysMap = new HashMap<>();
    private List<Consumer<Boolean>> loadingListeners = new ArrayList<>();
    private List<String> lastSelection = new ArrayList<>();

    public CatalogListViewDisplay(SumusBox box) {
        super();
        this.box = box;
    }

    @Override
    public void view(ElementView view) {
        this.view = view;
    }

    @Override
    public void provider(CatalogViewDisplayProvider provider) {
        this.provider = provider;
    }

    public void selectSorting(Sorting sorting) {
        this.sorting = sortingOf(sorting);
        sendSelectedSorting();
        sendClear();
        page(0);
    }

    @Override
    public int countItems() {
        return provider.countRecords(condition);
    }

    public void page(Integer value) {
        super.page(value);
    }

    @Override
    public void onOpenItem(Consumer<OpenItemEvent> listener) {
        openItemListeners.add(listener);
    }

    @Override
    public void onOpenItemDialog(Consumer<OpenItemDialogEvent> listener) {
        openItemDialogListeners.add(listener);
    }

    @Override
    public void onExecuteItemTask(Consumer<ExecuteItemTaskEvent> listener) {
        executeItemTaskListeners.add(listener);
    }

    @Override
    public void onCreateCluster(Consumer<io.intino.sumus.graph.Cluster> listener) {
        createClusterListeners.add(listener);
    }

    @Override
    public void reset() {
    }

    @Override
    public void onLoading(Consumer<Boolean> listener) {
        loadingListeners.add(listener);
    }

    @Override
    public ElementView view() {
        return view;
    }

    public void filter(String value) {
        this.condition = value;
        this.refresh();
    }

    @Override
    public void refresh(RecordItem... items) {
        Stream.of(items).forEach(this::refresh);
    }

    public void selectItems(String[] records) {
        List<String> newRecords = new ArrayList<>(Arrays.asList(records));
        newRecords.removeAll(lastSelection);
        newRecords.forEach(record -> {
            if (!recordDisplaysMap.containsKey(record)) renderDisplays(record);
            renderExpandedPictures(record);
        });
        this.lastSelection = new ArrayList<>(Arrays.asList(records));
    }

    public void openItem(String value) {
        if (view.onClickRecordEvent() == null) {
            if (provider.expandedStamps(view.mold()).size() > 0)
                notifier.refreshSelection(lastSelection.contains(value) ? emptyList() : singletonList(value));
            return;
        }

        if (view.onClickRecordEvent().openPanel() != null)
            notifyOpenItem(value);
        else if (view.onClickRecordEvent().openDialog() != null)
            notifyOpenDialog(value);
    }

    private void notifyOpenItem(String item) {
        openItemListeners.forEach(l -> l.accept(new OpenItemEvent() {
            @Override
            public String item() {
                return item;
            }

            @Override
            public String label() {
                Optional<Mold.Block.Stamp> titleStamp = provider.stamps(view.mold()).stream().filter(s -> s.i$(Title.class)).findAny();
                return titleStamp.isPresent() ? titleStamp.get().a$(Title.class).title(record()) : record().name$();
            }

            @Override
            public Record record() {
                return provider.record(new String(Base64.getDecoder().decode(item)));
            }

            @Override
            public Panel panel() {
                return view.onClickRecordEvent().openPanel().panel();
            }

            @Override
            public NameSpace nameSpace() {
                Optional<User> user = provider.user();
                return provider.nameSpaceHandler().selectedNameSpace(user.isPresent() ? user.get().username() : null);
            }

            @Override
            public TimeRange range() {
                return provider.range();
            }

            @Override
            public Tree breadcrumbs() {
                Optional<User> user = provider.user();
                OnClickRecord.OpenPanel openPanel = view.onClickRecordEvent().openPanel();
                return openPanel != null ? openPanel.breadcrumbs(record(), user.isPresent() ? user.get().username() : null) : null;
            }
        }));
    }

    private void notifyOpenDialog(String item) {
        openItemDialogListeners.forEach(l -> l.accept(new OpenItemDialogEvent() {
            @Override
            public String item() {
                return item;
            }

            @Override
            public String path() {
                return view.onClickRecordEvent().openDialog().path(item);
            }

            @Override
            public int width() {
                return view.onClickRecordEvent().openDialog().width();
            }

            @Override
            public int height() {
                return view.onClickRecordEvent().openDialog().height();
            }
        }));
    }

    public void renderExpandedPictures() {
        lastSelection.forEach(this::renderExpandedPictures);
    }

    private void renderDisplays(String item) {
        Map<Mold.Block.Display, StampDisplay> displays = displays(item);
        displays.forEach((stamp, display) -> {
            add(display);
            display.personify(item + stamp.name$());
            display.refresh();
        });
        recordDisplaysMap.put(item, displays.values().stream().collect(toList()));
    }

    private void renderExpandedPictures(String item) {
        refreshPictures(item, expandedPictures(item));
    }

    private void refreshPictures(String item) {
        refreshPictures(item, allPictures(item));
    }

    private void refreshPictures(String item, List<Picture> pictures) {
        Record record = provider.record(new String(Base64.getDecoder().decode(item)));
        pictures.forEach(stamp -> {
            try {
                String name = stamp.name$();
                Object data = stamp.internalValue(record, provider.user().isPresent() ? provider.user().get().username() : null);
                if ((! (data instanceof List)) || ((List) data).size() != 1) return;
                List<URL> values = (List<URL>)data;
                byte[] pictureBytes = IOUtils.toByteArray(values.get(0).openStream());
                byte[] picture = Base64.getEncoder().encode(pictureBytes);
                notifier.refreshPicture(PictureDataBuilder.build(record, name, "data:image/png;base64," + new String(picture)));
            } catch (IOException e) {
                e.printStackTrace();
            }
        });
    }

    public void createClusterGroup(ClusterGroup value) {
        String username = session() != null && session().user() != null ? session().user().username() : null;
        List<Entity> entities = value.entities().stream().map(entityName -> provider.record(new String(Base64.getDecoder().decode(entityName))).a$(Entity.class)).collect(toList());
        Cluster cluster = registerClusterGroup(box, view.element(), value.cluster(), value.name(), entities, username);
        createClusterListeners.forEach(l -> l.accept(cluster));
    }

    @Override
    protected void sendItems(int start, int limit) {
        notifier.refresh(RecordItemBuilder.buildList(provider.records(start, limit, condition, sorting), recordItemDisplayProvider(provider, view), baseAssetUrl(session())));
    }

    @Override
    protected void sendClear() {
        notifier.clear();
    }

    @Override
    protected void sendPageSize(int pageSize) {
        notifier.refreshPageSize(pageSize);
    }

    @Override
    protected void sendCount(int count) {
        notifier.refreshCount(count);
    }

    @Override
    protected void init() {
        super.init();

        List<Catalog.Analysis.Sorting> sortings = provider.sortings();
        this.sorting = sortings.size() > 0 ? sortingOf(sortings.get(0)) : null;

        sendView();
        sendSortingList(sortings);
        sendSelectedSorting();
    }

    @Override
    public void refresh() {
        notifyLoading(true);
        super.refresh();
        notifyLoading(false);
    }

    private void sendView() {
        notifier.refreshView(ElementViewBuilder.build(view));
    }

    private void sendSortingList(List<Catalog.Analysis.Sorting> sortings) {
        notifier.refreshSortingList(CatalogSortingBuilder.buildList(sortings));
    }

    private void sendSelectedSorting() {
        if (sorting == null) return;
        notifier.refreshSelectedSorting(CatalogSortingBuilder.build(sorting));
    }

    private Map<Mold.Block.Display, StampDisplay> displays(String record) {
        List<io.intino.sumus.graph.Mold.Block.Stamp> stamps = provider.stamps(view.mold()).stream().filter(s -> s.i$(Mold.Block.Display.class)).collect(toList());
        Map<Mold.Block.Display, StampDisplay> nullableMap = stamps.stream().collect(Collectors.toMap(s -> s.a$(Mold.Block.Display.class), stamp -> provider.display(stamp.name$())));
        Map<Mold.Block.Display, StampDisplay> result = nullableMap.entrySet().stream().filter(e -> e.getValue() != null).collect(toMap(Map.Entry::getKey, Map.Entry::getValue));
        result.forEach((key, value) -> value.record(provider.record(record)));
        return result;
    }

    private List<Picture> expandedPictures(String record) {
        return provider.expandedStamps(view.mold()).stream().filter(s -> s.i$(Picture.class))
                .map(s -> s.a$(Picture.class))
                .collect(toList());
    }

    private List<Picture> allPictures(String record) {
        return provider.stamps(view.mold()).stream().filter(s -> s.i$(Picture.class))
                .map(s -> s.a$(Picture.class))
                .collect(toList());
    }

    private CatalogViewDisplayProvider.Sorting sortingOf(Sorting sorting) {
        return sortingOf(sorting.name(), sorting.mode());
    }

    private CatalogViewDisplayProvider.Sorting sortingOf(Catalog.Analysis.Sorting sorting) {
        return sortingOf(sorting.name$(), "Ascendant");
    }

    private CatalogViewDisplayProvider.Sorting sortingOf(String name, String mode) {
        return new CatalogViewDisplayProvider.Sorting() {
            @Override
            public String name() {
                return name;
            }

            @Override
            public Mode mode() {
                return Mode.valueOf(mode);
            }

            @Override
            public int comparator(Record record1, Record record2) {
                return provider.sorting(name).comparator(record1, record2);
            }
        };
    }

    private void refresh(RecordItem item) {
        notifier.refreshItem(item);
        if (recordDisplaysMap.containsKey(item.name()))
            recordDisplaysMap.get(item.name()).forEach(Display::refresh);
    }

    public void itemRefreshed(String record) {
        refreshPictures(record);
    }

    private void notifyLoading(boolean value) {
        loadingListeners.forEach(l -> l.accept(value));
    }

    public void openItemDialogOperation(OpenItemDialogParameters params) {
        openItemDialogListeners.forEach(l -> l.accept(openItemDialogEvent(params.item(), provider.stamp(view.mold(), params.stamp()))));
    }

    public void executeItemTaskOperation(ExecuteItemTaskParameters params) {
        executeItemTaskListeners.forEach(l -> l.accept(executeItemTaskEvent(params.item(), provider.stamp(view.mold(), params.stamp()))));
    }

    public io.intino.konos.server.activity.spark.ActivityFile downloadItemOperation(DownloadItemParameters value) {
        return null;
    }

    public void executeOperation(ElementOperationParameters value) {
        provider.executeOperation(value, selectedRecords());
    }

    public io.intino.konos.server.activity.spark.ActivityFile downloadOperation(ElementOperationParameters value) {
        Resource resource = provider.downloadOperation(value, selectedRecords());
        return new ActivityFile() {
            @Override
            public String label() {
                return resource.label();
            }

            @Override
            public InputStream content() {
                return resource.content();
            }
        };
    }

    private List<Record> selectedRecords() {
        return lastSelection.stream().map(name -> provider.record(new String(Base64.getDecoder().decode(name)))).collect(toList());
    }

    public void saveItem(SaveItemParameters value) {
        if (selectedRecords().size() != 1) return;
        provider.saveItem(value, selectedRecords().get(0));
    }

}