package io.intino.sumus;

import io.intino.sumus.helpers.UtcDateTime;
import io.intino.tara.io.Stash;
import io.intino.tara.magritte.stores.FileSystemStore;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.Clock;
import java.time.Instant;
import java.util.List;
import java.util.stream.Stream;

import static io.intino.sumus.datawarehouse.store.PathBuilder.DigestsSuffix;
import static io.intino.sumus.datawarehouse.store.PathBuilder.EventsSuffix;
import static java.io.File.separator;
import static java.util.stream.Collectors.toList;

public class SumusStore extends FileSystemStore {

	public SumusStore(File file) {
		super(file);
	}

	public File storeFolder(){
		return file;
	}

	@Override
	public Stash stashFrom(String path) {
		Stash stash = super.stashFrom(path);
		return stash == null ? defaultStash() : stash;
	}

	private Stash defaultStash() {
		Stash defaultStash = new Stash();
		defaultStash.language = "Sumus";
		return defaultStash;
	}

	public Instant minInstantForNameSpace(String nameSpace) {
		List<File> years = years(nameSpace);
		return findInLowerThanDayScale(years)
				.findFirst().orElseGet(() -> findInLowerThanYearScale(years).findFirst().orElse(Instant.now(Clock.systemUTC())));
	}

	public Instant maxInstantForNameSpace(String nameSpace) {
		List<File> years = years(nameSpace);
		return findInLowerThanDayScale(years)
				.reduce((l, r) -> r).orElseGet(() -> findInLowerThanYearScale(years).reduce((l, r) -> r).orElse(Instant.now(Clock.systemUTC())));
	}

	private Stream<Instant> findInLowerThanDayScale(List<File> years) {
		return years.stream()
				.map(File::toPath)
				.flatMap(this::list)
				.flatMap(this::list)
				.map(Path::toFile)
				.filter(this::isNotMacFile)
				.map(this::toInstant)
				.sorted();
	}

	private Stream<Instant> findInLowerThanYearScale(List<File> years) {
		return years.stream()
				.map(File::toPath)
				.flatMap(this::list)
				.map(Path::toFile)
				.filter(this::isNotMacFile)
				.map(this::toInstantUntilDay)
				.sorted();
	}

	private Instant toInstant(File file) {
		File dayFile = file.getParentFile();
		int day = Integer.valueOf(dayFile.getName());
		File yearFile = dayFile.getParentFile();
		int year = Integer.valueOf(yearFile.getName());
		String time = afterDash(file.getName());
		int hour = Integer.valueOf(time.substring(0, 2));
		int minutes = time.length() <= 2 ? 0 : Integer.valueOf(time.substring(2));
		return new UtcDateTime(year, 1, 1, hour, minutes).plusDays(day - 1).toInstant();
	}

	private Instant toInstantUntilDay(File file) {
		int day = Integer.valueOf(afterDash(file.getName()));
		File yearFile = file.getParentFile();
		int year = Integer.valueOf(yearFile.getName());
		return new UtcDateTime(year, 1, 1, 0, 0).plusDays(day - 1).toInstant();
	}

	private List<File> years(String nameSpace) {
		List<File> directories = list(new File(file, nameSpace + DigestsSuffix).toPath()).map(Path::toFile).filter(File::isDirectory).filter(this::isNotMacFile).collect(toList());
		directories.addAll(list(new File(file, nameSpace + EventsSuffix).toPath()).map(Path::toFile).filter(File::isDirectory).filter(this::isNotMacFile).collect(toList()));
		directories = directories.stream().map(d -> list(d.toPath())).flatMap(p -> p.map(Path::toFile)).filter(File::isDirectory).filter(this::isNotMacFile).collect(toList());
		return directories.stream().filter(f -> f.getName().matches("\\d+")).sorted(File::compareTo).collect(toList());
	}

	private boolean isNotMacFile(File file) {
		return !file.toString().contains(".DS_Store");
	}

	private Stream<Path> list(Path path) {
		try {
			return Files.list(path);
		} catch (IOException e) {
			return Stream.empty();
		}
	}

	private String afterDash(String fileName) {
		if (!fileName.contains("-"))
			return fileName;

		fileName = fileName.substring(fileName.indexOf("-") + 1, fileName.lastIndexOf("."));
		if (!fileName.contains("-"))
			return fileName;

		return fileName.substring(0, fileName.indexOf("-"));
	}

	public String absoluteFilename(String storeFilename) {
		return file.getAbsoluteFile() + separator + storeFilename;
	}
}