package io.intino.sumus.box.displays.builders;

import io.intino.sumus.box.schemas.Mold;
import io.intino.sumus.box.schemas.MoldBlock;
import io.intino.sumus.box.schemas.Property;
import io.intino.sumus.box.schemas.Stamp;
import io.intino.sumus.graph.Mold.Block.*;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

import static java.util.stream.Collectors.toList;

public class MoldBuilder {

    public static Mold build(io.intino.sumus.graph.Mold mold) {
        return new Mold().moldBlockList(mold.blockList().stream().map(MoldBuilder::buildBlock).collect(toList()));
    }

    private static MoldBlock buildBlock(io.intino.sumus.graph.Mold.Block block) {
        return new MoldBlock().name(block.name$()).style(block.style()).expanded(block.isExpanded())
                              .hiddenIfMobile(block.hiddenIfMobile())
                              .layout(block.layout().stream().map(Enum::toString).collect(Collectors.joining(" ")))
                              .width(block.width())
                              .height(block.height())
                              .moldBlockList(block.blockList().stream().map(MoldBuilder::buildBlock).collect(toList()))
                              .stampList(block.stampList().stream().map(MoldBuilder::buildStamp).collect(toList()));
    }

    private static Stamp buildStamp(io.intino.sumus.graph.Mold.Block.Stamp stamp) {
        Stamp result = new Stamp().name(stamp.name$()).label(stamp.label())
                                  .editable(stamp.isEditable())
                                  .shape(shapeOf(stamp)).layout(stamp.layout().toString())
                                  .height(stamp.height());

        List<Property> propertyList = new ArrayList<>();
        addCommonProperties(propertyList, stamp);
        addRatingProperties(propertyList, stamp);
        addEmbeddedCatalogProperties(propertyList, stamp);
        addIconProperties(propertyList, stamp);
        addDownloadOperationProperties(propertyList, stamp);
        addExportOperationProperties(propertyList, stamp);
        result.propertyList(propertyList);

        return result;
    }

    private static String shapeOf(io.intino.sumus.graph.Mold.Block.Stamp stamp) {
        if (stamp.i$(Title.class)) return "title";
        if (stamp.i$(Description.class)) return "description";
        if (stamp.i$(Icon.class)) return "icon";
        if (stamp.i$(Rating.class)) return "rating";
        if (stamp.i$(Highlight.class)) return "highlight";
        if (stamp.i$(Picture.class)) return "picture";
        if (stamp.i$(OpenDialogOperation.class)) return "open-dialog-operation";
        if (stamp.i$(DownloadOperation.class)) return "download-operation";
        if (stamp.i$(ExportOperation.class)) return "export-operation";
        if (stamp.i$(TaskOperation.class)) return "task-operation";
        if (stamp.i$(Page.class)) return "page";
        if (stamp.i$(Display.class)) return "display";
        if (stamp.i$(Location.class)) return "location";
        if (stamp.i$(Breadcrumbs.class)) return "breadcrumbs";
        if (stamp.i$(RecordLinks.class)) return "record-links";
        if (stamp.i$(CatalogLink.class)) return "catalog-link";
        if (stamp.i$(Snippet.class)) return "snippet";
        if (stamp.i$(EmbeddedCatalog.class)) return "embedded-catalog";
        return "";
    }

    private static void addCommonProperties(List<Property> propertyList, io.intino.sumus.graph.Mold.Block.Stamp stamp) {
        if (!stamp.suffix().isEmpty()) propertyList.add(shapeProperty("suffix", stamp.suffix()));
        if (!stamp.defaultStyle().isEmpty()) propertyList.add(shapeProperty("defaultStyle", stamp.defaultStyle()));
    }

    private static void addRatingProperties(List<Property> propertyList, io.intino.sumus.graph.Mold.Block.Stamp stamp) {
        if (!stamp.i$(Rating.class)) return;
        propertyList.add(shapeProperty("icon", stamp.a$(Rating.class).ratingIcon()));
    }

    private static void addEmbeddedCatalogProperties(List<Property> propertyList, io.intino.sumus.graph.Mold.Block.Stamp stamp) {
        if (!stamp.i$(EmbeddedCatalog.class)) return;
        propertyList.add(shapeProperty("catalog", stamp.a$(EmbeddedCatalog.class).catalog().name$()));
    }

    private static void addIconProperties(List<Property> propertyList, io.intino.sumus.graph.Mold.Block.Stamp stamp) {
        if (!stamp.i$(Icon.class)) return;
        propertyList.add(shapeProperty("icon-type", stamp.i$(SumusIcon.class) ? "sumus" : ""));
    }

    private static void addDownloadOperationProperties(List<Property> propertyList, io.intino.sumus.graph.Mold.Block.Stamp stamp) {
        if (!stamp.i$(DownloadOperation.class)) return;
        DownloadOperation downloadStamp = stamp.a$(DownloadOperation.class);
        propertyList.add(shapeProperty("title", downloadStamp.title()));
        propertyList.add(shapeProperty("options", String.join(",", downloadStamp.options())));
    }

    private static void addExportOperationProperties(List<Property> propertyList, io.intino.sumus.graph.Mold.Block.Stamp stamp) {
        if (!stamp.i$(ExportOperation.class)) return;
        ExportOperation exportStamp = stamp.a$(ExportOperation.class);
        propertyList.add(shapeProperty("title", exportStamp.title()));
        propertyList.add(shapeProperty("options", String.join(",", exportStamp.options())));
        propertyList.add(shapeProperty("from", String.valueOf(exportStamp.from().toEpochMilli())));
        propertyList.add(shapeProperty("to", String.valueOf(exportStamp.to().toEpochMilli())));
    }

    private static Property shapeProperty(String name, String value) {
        return new Property().name(name).value(value);
    }

}
