package io.intino.sumus.box.displays.builders;

import io.intino.sumus.Category;
import io.intino.sumus.box.schemas.Catalog;
import io.intino.sumus.box.schemas.Group;
import io.intino.sumus.box.schemas.Grouping;
import io.intino.sumus.box.schemas.Sorting;
import io.intino.sumus.graph.Categorization;
import io.intino.sumus.graph.Cluster;
import io.intino.sumus.queries.CatalogManager;

import java.util.List;
import java.util.Objects;

import static java.util.Collections.emptyList;
import static java.util.stream.Collectors.toList;

public class CatalogBuilder {

    public static Catalog build(io.intino.sumus.graph.Catalog catalog, CatalogManager manager, String label, boolean embedded) {
        return new Catalog().name(catalog.name$()).label(label)
                .embedded(embedded)
                .hideGroupings(catalog.analysis() == null || catalog.analysis().groupingList().size() <= 0)
                .groupingList(buildGroupingList(catalog, manager))
                .sortingList(buildSortingList(catalog));
    }

    private static List<Grouping> buildGroupingList(io.intino.sumus.graph.Catalog catalog, CatalogManager manager) {
        if (catalog.analysis() == null) return emptyList();
        return catalog.analysis().abstractGroupingList().stream().map(g -> buildGrouping(g, manager)).filter(Objects::nonNull).collect(toList());
    }

    private static Grouping buildGrouping(io.intino.sumus.graph.Catalog.Analysis.AbstractGrouping grouping, CatalogManager manager) {
        List<Group> groupList = buildGroupList(grouping, manager);
        int countItems = groupList.stream().mapToInt(Group::count).sum();

        if (countItems <= 0) return null;

        return new Grouping().name(grouping.name$()).label(grouping.label()).histogram(grouping.histogram().toString())
                .type(typeOf(grouping))
                .groupList(groupList)
                .countItems(countItems);
    }

    private static List<Sorting> buildSortingList(io.intino.sumus.graph.Catalog catalog) {
        if (catalog.analysis() == null) return emptyList();
        return CatalogSortingBuilder.buildList(catalog.analysis().sortingList());
    }

    private static String typeOf(io.intino.sumus.graph.Catalog.Analysis.AbstractGrouping grouping) {
        return grouping.i$(io.intino.sumus.graph.Catalog.Analysis.Grouping.class) ? "Grouping" : "ClusterGrouping";
    }

    private static List<Group> buildGroupList(io.intino.sumus.graph.Catalog.Analysis.AbstractGrouping grouping, CatalogManager manager) {
        if (grouping.i$(io.intino.sumus.graph.Catalog.Analysis.Grouping.class)) {
            Categorization categorization = grouping.a$(io.intino.sumus.graph.Catalog.Analysis.Grouping.class).categorization();
            return buildCategorizationGroupList(manager.categories(categorization).toList(), manager.filteredCategories(categorization));
        }

        if (grouping.i$(io.intino.sumus.graph.Catalog.Analysis.ClusterGrouping.class))
            return buildClusterGroupList(grouping.a$(io.intino.sumus.graph.Catalog.Analysis.ClusterGrouping.class).cluster().groupList());

        return emptyList();
    }

    private static List<Group> buildCategorizationGroupList(List<Category> categories, List<String> selected) {
        return categories.stream().map(c -> buildGroup(c, selected.contains(c.label()))).collect(toList());
    }

    private static Group buildGroup(Category category, boolean selected) {
        return new Group().name(category.name()).label(category.label()).selected(selected).count(category.recordIds().size());
    }

    private static List<Group> buildClusterGroupList(List<Cluster.Group> groups) {
        return groups.stream().map(CatalogBuilder::buildGroup).collect(toList());
    }

    private static Group buildGroup(Cluster.Group group) {
        return new Group().name(group.name$()).label(group.label()).count(group.entities().size());
    }

}
