package io.intino.sumus.box.displays;

import io.intino.konos.server.activity.displays.DisplayNotifier;
import io.intino.sumus.RecordList;
import io.intino.sumus.analytics.TimeRange;
import io.intino.sumus.box.SumusBox;
import io.intino.sumus.box.displays.providers.CatalogViewDisplayProvider;
import io.intino.sumus.box.schemas.GroupingSelection;
import io.intino.sumus.graph.*;
import io.intino.sumus.graph.rules.Mode;
import io.intino.sumus.graph.rules.TimeScale;
import io.intino.sumus.helpers.TimeScaleHandler;
import io.intino.sumus.queries.CatalogManager;
import io.intino.sumus.queries.TemporalRecordQuery;
import io.intino.tara.magritte.Concept;

import java.time.Instant;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static java.util.stream.Collectors.toList;

public abstract class TemporalCatalogDisplay<DN extends DisplayNotifier, N extends ChartNavigatorDisplay> extends CatalogDisplay<DN> implements CatalogViewDisplayProvider {
	private N navigatorDisplay = null;

	public TemporalCatalogDisplay(SumusBox box, N navigatorDisplay) {
		super(box);
		this.navigatorDisplay = navigatorDisplay;
	}

	public void selectNameSpaceAndRange(NameSpace nameSpace, TimeRange range) {
		dirty(true);
		nameSpaceHandler().select(nameSpace);
		TimeScale scale = range.scale();
		TimeScale referenceScale = (scale.ordinal() > TimeScale.Day.ordinal()) ? scale : TimeScale.Day;
		timeScaleHandler().updateRange(range.from(), referenceScale.addTo(range.to(), 1), false);
	}

	@Override
	protected void selectNameSpace(NameSpace nameSpace) {
		dirty(true);
		resetViews();
		this.refresh();
		reloadGroupings();
	}

	private void resetViews() {
		child(CatalogViewListDisplay.class).viewList().forEach(CatalogViewDisplay::reset);
	}

	@Override
	public StampDisplay display(String stamp) {
		if (box.stampDisplayProvider() == null)
			return null;

		TemporalStampDisplay display = box.stampDisplayProvider().temporalDisplay(stamp);
		display.nameSpace(nameSpace());
		display.range(timeScaleHandler().range());

		return display;
	}

	public void selectGrouping(GroupingSelection value) {
		super.selectGrouping(value);
	}

	@Override
	public void refreshView() {
		super.refreshView();

		currentView().ifPresent(catalogView -> {
			AbstractView view = views().stream().filter(v -> v.name$().equals(catalogView.view().name())).findFirst().orElse(null);
			if (view != null && view.i$(Catalog.OlapView.class))
				hideTimeNavigator();
			else
				showTimeNavigator();
		});
	}

	@Override
	public TimeRange range() {
		return timeScaleHandler().range();
	}

	@Override
	public TimeScale scale() {
		return range().scale();
	}

	@Override
	public void refresh() {
		resetGrouping();
		createCatalogManager();
		refreshView();
	}

	@Override
	public Record rootRecord(List<Record> recordList) {
		return element().asTemporalHolder().rootRecord(recordList, username(), nameSpace(), range());
	}

	@Override
	public Record defaultRecord(String id) {
		return element().asTemporalHolder().defaultRecord(id, username(), nameSpace(), range());
	}

	@Override
	protected void createCatalogManager() {
		RecordList<TemporalRecord> recordList = queryEngine().temporalRecords(query(null));
		manager = new CatalogManager(recordList.items().stream().map(i -> i.a$(Record.class)).collect(toList()), categorizations(), keys());
	}

	@Override
	protected <R extends Record> RecordList<R> records(String condition) {
		TemporalRecordQuery query = query(condition);
		if (!equalsRange()) resetGrouping();
		RecordList<R> recordList = (RecordList<R>) queryEngine().temporalRecords(query);
		applyFilter(recordList);
		if (!equalsRange()) reloadGroupings();
		range(timeScaleHandler().range());
		filterTimezone(recordList, range());
		return recordList;
	}

	@Override
	protected Record loadRecord(String id) {
		return queryEngine().temporalRecord(id, username());
	}

	@Override
	protected boolean canCreateClusters() {
		return false;
	}

	@Override
	protected void init() {
		TimeScaleHandler timeScaleHandler = buildTimeScaleHandler();
		buildNavigatorDisplay(timeScaleHandler);
		super.init();
	}

	protected void refresh(Instant instant) {
		refresh();
	}

	protected void refresh(TimeRange range) {
		refresh();
	}

	private void buildNavigatorDisplay(TimeScaleHandler timeScaleHandler) {
		navigatorDisplay.timeScaleHandler(timeScaleHandler);
		configureNavigatorDisplay(navigatorDisplay, timeScaleHandler);
		add(navigatorDisplay);
		navigatorDisplay.personify();
	}

	private TimeScaleHandler buildTimeScaleHandler() {
		TimeRange range = queryEngine().temporalRecordRange(nameSpaceHandler().nameSpaces(username()));
		TimeScaleHandler.Bounds bounds = new TimeScaleHandler.Bounds();
		List<TimeScale> scales = element().asTemporalHolder().scales();
		Map<TimeScale, io.intino.sumus.helpers.Bounds.Zoom> zoomMap = new HashMap<>();

		bounds.range(new TimeRange(range.from(), range.to(), TimeScale.Minute));
		bounds.mode(Mode.ToTheLast);

		scales.forEach(scale -> zoomMap.put(scale, new io.intino.sumus.helpers.Bounds.Zoom().min(1).max(maxZoom())));
		bounds.zooms(zoomMap);

		TimeScaleHandler timeScaleHandler = new TimeScaleHandler(bounds, scales, scales.get(0));
		timeScaleHandler.availableScales(localizedScales(scales));
		configureTimeScaleHandler(timeScaleHandler, range, scales);

		return timeScaleHandler;
	}

	@Override
	public boolean dirty() {
		return super.dirty() || !equalsRange();
	}

	protected boolean equalsRange() {
		TimeRange range = timeScaleHandler().range();
		return range() != null && range().from() == range.from() && range().scale() == range.scale();
	}

	protected abstract int maxZoom();
	protected abstract void configureTimeScaleHandler(TimeScaleHandler timeScaleHandler, TimeRange range, List<TimeScale> scales);
	protected abstract void configureNavigatorDisplay(N navigatorDisplay, TimeScaleHandler timeScaleHandler);
	protected abstract Concept recordConcept();
	protected abstract void hideTimeNavigator();
	protected abstract void showTimeNavigator();
	protected abstract void addTime(TemporalRecordQuery.Builder builder);
	protected abstract <R extends Record> void filterTimezone(RecordList<R> recordList, TimeRange range);
	protected abstract TimeScaleHandler timeScaleHandler();

	private TemporalRecordQuery query(String condition) {
		TemporalRecordQuery.Builder builder = new TemporalRecordQuery.Builder();
		builder.condition(condition);
		builder.nameSpace(nameSpace());
		addTime(builder);
		return builder.build(recordConcept(), username());
	}

	private void reloadGroupings() {
		sendCatalog();
	}

}