package io.intino.sumus.box.displays;

import io.intino.sumus.box.SumusBox;
import io.intino.sumus.box.displays.ElementViewDisplay.ExecuteItemTaskEvent;
import io.intino.sumus.box.displays.ElementViewDisplay.OpenItemDialogEvent;
import io.intino.sumus.box.displays.ElementViewDisplay.OpenItemEvent;
import io.intino.sumus.box.displays.builders.ItemBuilder;
import io.intino.sumus.box.displays.notifiers.PanelDisplayNotifier;
import io.intino.sumus.box.displays.providers.ElementViewDisplayProvider;
import io.intino.sumus.box.schemas.Item;
import io.intino.sumus.graph.*;

import java.util.HashMap;
import java.util.Map;
import java.util.function.Function;

import static java.util.stream.Collectors.toList;

public class PanelDisplay extends ElementDisplay<Panel, PanelDisplayNotifier> implements ElementViewDisplayProvider {
	protected Map<Class<? extends ElementRender>, Function<ElementRender, PanelViewDisplay>> displayBuilders = new HashMap<>();
	private Map<String, PanelViewDisplay> viewDisplayMap = new HashMap<>();
	private static final String ViewId = "%s%s";

	public PanelDisplay(SumusBox box) {
		super(box);
		registerBuilders();
	}

	@Override
	protected void refreshBreadcrumbs(String breadcrumbs) {
		notifier.refreshBreadcrumbs(breadcrumbs);
	}

	@Override
	protected void createPanel(String item) {
		notifier.createPanel(item);
	}

	@Override
	protected void showPanel() {
		notifier.showPanel();
	}

	@Override
	protected void hidePanel() {
		notifier.hidePanel();
	}

	@Override
	protected void showDialog() {
		notifier.showDialog();
	}

	@Override
	protected void currentRecord(String id) {
	}

	@Override
	protected Record currentRecord() {
		return this.target();
	}

	@Override
	protected void notifyFiltered(boolean value) {
	}

	@Override
	protected void selectNameSpace(NameSpace nameSpace) {
	}

	@Override
	protected Record loadRecord(String id) {
		return queryEngine().record(id, username());
	}

	@Override
	protected void init() {
		super.init();
		sendTarget();
		sendViewList();
		createDialogContainer();
		if (views().size() > 0)
			selectView(views().get(0).name$());
	}

	private void sendTarget() {
		if (target() == null) return;
		notifier.refreshTarget(target().name$());
	}

	private void sendViewList() {
		notifier.refreshViewList(views().stream().map(this::itemOf).collect(toList()));
	}

	private Item itemOf(AbstractView view) {
		return ItemBuilder.build(view);
	}

	private void registerBuilders() {
		displayBuilders.put(RenderMold.class, this::buildCustomViewDisplay);
		displayBuilders.put(RenderCatalog.class, this::buildCatalogViewDisplay);
		displayBuilders.put(RenderOlap.class, this::buildOlapViewDisplay);
	}

	private PanelCustomViewDisplay buildCustomViewDisplay(ElementRender render) {
		return new PanelCustomViewDisplay(box);
	}

	private PanelCatalogViewDisplay buildCatalogViewDisplay(ElementRender render) {
		return new PanelCatalogViewDisplay(box);
	}

	private PanelOlapViewDisplay buildOlapViewDisplay(ElementRender render) {
		return new PanelOlapViewDisplay(box);
	}

	public void selectView(String name) {
		buildView(name).refresh();
	}

	private PanelViewDisplay buildView(String name) {
		if (viewDisplayMap.containsKey(name)) viewDisplayMap.get(name);
		notifyLoading(true);
		PanelViewDisplay display = buildView(views().stream().filter(v -> v.name$().equals(name)).findFirst().orElse(null));
		viewDisplayMap.put(name, display);
		notifyLoading(false);
		return display;
	}

	private PanelViewDisplay buildView(AbstractView view) {
		Panel.View panelView = view.a$(Panel.View.class);
		ElementRender render = panelView.elementRender();
		PanelViewDisplay display = displayBuilders.get(render.getClass()).apply(render);
		display.provider(this);
		display.view(panelView);
		display.context(element());
		display.target(target());
		display.onLoading(v -> notifyLoading((Boolean) v));
		display.onOpenItem(params -> openItem((OpenItemEvent) params));
		display.onOpenItemDialog(params -> openItemDialog((OpenItemDialogEvent) params));
		display.onExecuteItemTask(params -> executeItemTask((ExecuteItemTaskEvent) params));
		add(display);
		display.personifyOnce(viewId(view));
		return display;
	}

	private String viewId(AbstractView view) {
		if (target() == null) return view.name$();
		return String.format(ViewId, target().name$(), view.name$());
	}

	public void navigate(String key) {
		super.navigate(key);
	}

}