package io.intino.sumus.box.displays;

import io.intino.konos.server.activity.displays.DisplayNotifier;
import io.intino.sumus.box.SumusBox;
import io.intino.sumus.graph.*;
import io.intino.sumus.graph.entityholder.EntityHolderCatalog;
import io.intino.sumus.graph.eventholder.EventHolderCatalog;
import io.intino.sumus.graph.reportholder.ReportHolderCatalog;
import io.intino.sumus.helpers.NameSpaceHandler;
import io.intino.tara.magritte.Layer;

import java.util.HashMap;
import java.util.Map;
import java.util.function.Function;

public abstract class ElementRepositoryDisplay<DN extends DisplayNotifier> extends SumusDisplay<DN> implements ElementDisplayManager {
    private final NameSpaceHandler nameSpaceHandler;
    private Map<Class<? extends Layer>, Function<Element, ElementDisplay>> builders = new HashMap<>();
    private Map<String, ElementDisplay> displayMap = new HashMap<>();

    public ElementRepositoryDisplay(SumusBox box) {
        super(box);
        registerBuilders();
        this.nameSpaceHandler = new NameSpaceHandler(box.graph());
    }

    public <E extends ElementDisplay> E openElement(String label) {
        Element element = element(label);
        Record target = target(label);
        E display = display(label);

        if (display != null) {
            display.clearFilter();
            refreshSelected(label);
            return display;
        }

        refreshLoading(true);
        display = addAndBuildDisplay(element, target, label);
        refreshLoaded();
        refreshSelected(label);

        return display;
    }

    public <E extends ElementDisplay> E createElement(Element element, Record target) {
        E display = display(target.name$());
        if (display != null) return display;
        return buildDisplay(element, target, target.name$());
    }

    public void selectNameSpace(String key) {
        nameSpaceHandler.select(key);
    }

    @Override
    public <E extends ElementDisplay> E display(String label) {
        return (E) displayMap.getOrDefault(label, null);
    }

    protected abstract void refreshSelected(String label);
    protected abstract void refreshLoading(boolean withMessage);
    protected abstract void refreshLoaded();
    protected abstract Element element(String label);
    protected abstract Record target(String label);

    protected Class classFor(Element element) {
        if (element.i$(Catalog.class))
            return element.a$(Catalog.class).asHolder().getClass();
        return element.a$(Panel.class).getClass();
    }

    protected NameSpaceHandler nameSpaceHandler() {
        return nameSpaceHandler;
    }

    private void registerBuilders() {
        builders.put(Panel.class, this::buildPanelDisplay);
        builders.put(EntityHolderCatalog.class, this::buildEntityCatalogDisplay);
        builders.put(EventHolderCatalog.class, this::buildEventCatalogDisplay);
        builders.put(ReportHolderCatalog.class, this::buildReportCatalogDisplay);
    }

    private PanelDisplay buildPanelDisplay(Element component) {
        PanelDisplay display = new PanelDisplay(box);
        display.element(component.a$(Panel.class));
        return display;
    }

    private CatalogDisplay buildEntityCatalogDisplay(Element component) {
        EntityCatalogDisplay display = new EntityCatalogDisplay(box);
        display.element(component.a$(Catalog.class));
        return display;
    }

    private CatalogDisplay buildEventCatalogDisplay(Element component) {
        EventCatalogDisplay display = new EventCatalogDisplay(box);
        display.element(component.a$(Catalog.class));
        return display;
    }

    private CatalogDisplay buildReportCatalogDisplay(Element component) {
        ReportCatalogDisplay display = new ReportCatalogDisplay(box);
        display.element(component.a$(Catalog.class));
        return display;
    }

    private <E extends ElementDisplay> E addAndBuildDisplay(Element element, Record target, String label) {
        E display = buildDisplay(element, target, label);
        display.personifyOnce(label);
        return display;
    }

    private <E extends ElementDisplay> E buildDisplay(Element element, Record target, String label) {
        E display = buildDisplayFor(element, target, label);
        display.elementDisplayManager(this);
        add(display);

        display.onLoading((value) -> {
            if ((Boolean)value) refreshLoading(false);
            else refreshLoaded();
        });

        return display;
    }

    private <E extends ElementDisplay> E buildDisplayFor(Element element, Record target, String label) {
        Class clazz = classFor(element);

        ElementDisplay display = builders.get(clazz).apply(element);
        display.nameSpaceHandler(nameSpaceHandler);
        display.label(label);
        display.element(element);
        display.target(target);
        displayMap.put(label, display);

        return (E) display;
    }

}