import React from "react";
import { withStyles } from '@material-ui/core/styles';
import AbstractCubeBarChart from "../../gen/displays/AbstractCubeBarChart";
import CubeBarChartNotifier from "../../gen/displays/notifiers/CubeBarChartNotifier";
import CubeBarChartRequester from "../../gen/displays/requesters/CubeBarChartRequester";
import DisplayFactory from 'alexandria-ui-elements/src/displays/DisplayFactory';
import { withSnackbar } from 'notistack';
import Highcharts from 'highcharts';
import HighchartsMore from 'highcharts/highcharts-more'
import HighchartsReact from 'highcharts-react-official';
import Select from "react-select";
import 'alexandria-ui-elements/res/styles/layout.css';
import { RiseLoader } from "react-spinners";
import Theme from "app-elements/gen/Theme";

HighchartsMore(Highcharts);

const styles = theme => ({
    selector : {
        width: '250px',
        marginRight: '10px'
    },
    sortText : {
        fontSize: '9pt',
        marginRight:'10px',
        fontWeight: 'bold',
    },
    sortLink : {
        fontSize: '9pt',
        marginRight:'10px',
        cursor: 'pointer',
        color: '#005b9f',
    },
});

class CubeBarChart extends AbstractCubeBarChart {

	constructor(props) {
		super(props);
		this.notifier = new CubeBarChartNotifier(this);
		this.requester = new CubeBarChartRequester(this);
		this.loadedSeries = {};
		this.state = {
		    info: { indicators: [], drillCategories: [], divideByCategories: [], sort: null, mode: "Olap" },
		    toolbar : { visible: true },
		    series : [],
		    selection : { indicator: null, drillCategory: null, divideByCategory: null },
		    ...this.state
		}
	};

    render() {
        return (
            <div style={{height:'100%', width:'calc(100% - 20px)'}}>
                {this.state.info.mode == "Olap" && this.renderToolbar()}
                {this.state.loading && this.renderLoading()}
                {!this.state.loading && this.renderSections()}
            </div>
        );
    };

    renderToolbar = () => {
        const { classes } = this.props;
        const sort = this.state.info.sort;
        return (
            <div className="layout horizontal wrap" style={{height:'100%',marginBottom:'10px',marginLeft:'1px'}}>
				<Select isMulti={false} isSearchable className={classes.selector}
						placeholder="Seleccione un indicador" options={this.optionsOf(this.state.info.indicators)}
						value={this.optionOf(this.state.selection.indicator)} onChange={this.handleSelectIndicator.bind(this)}/>
				<div className="layout horizontal flex end-justified" style={{marginRight:'152px'}}>
				    <div style={{marginRight:'10px',fontSize:'9pt'}}>{this.translate("Sort by")}</div>
				    {(sort != null && sort !== "name") && <a className={classes.sortLink} onClick={this.handleSort.bind(this, "name")}>{this.translate("group")}</a>}
				    {(sort == null || sort === "name") && <div className={classes.sortText}>{this.translate("group")}</div>}
				    {this.state.info.sort !== "greater" && <a onClick={this.handleSort.bind(this, "greater")} className={classes.sortLink}>{this.translate("greater value")}</a>}
				    {sort === "greater" && <div className={classes.sortText}>{this.translate("greater value")}</div>}
				    {this.state.info.sort !== "lower" && <a onClick={this.handleSort.bind(this, "lower")} className={classes.sortLink}>{this.translate("lower value")}</a>}
				    {sort === "lower" && <div className={classes.sortText}>{this.translate("lower value")}</div>}
				</div>
            </div>
        );
    };

    renderLoading = () => {
        const { theme } = this.props;
        return (<div style={{position:'absolute',top:'50%',left:'35%'}}><RiseLoader color={theme.palette.secondary.main} loading={true}/></div>);
    };

    renderSections = () => {
        return this.state.series.map(serie => this.renderSection(serie));
    };

    renderSection = (serie) => {
        const hasDivideByCategories = this.state.info.divideByCategories.length > 0;
        return (
            <React.Fragment>
                {hasDivideByCategories && <div style={{backgroundColor:serie.backgroundColor,color:serie.color,fontSize:serie.fontSize+"pt",textAlign:'center',padding:'6px 16px 6px 16px'}}>{serie.name}</div>}
                {this.renderBar(serie)}
            </React.Fragment>
        );
    };

    renderBar = (serie) => {
        if (this.state.selection.indicator == null) return (<React.Fragment/>);
        const handleSelectDrillCategory = this.handleSelectDrillCategory.bind(this);
        const display = this;
        const options = {
            chart: {
                type: 'bar',
                height: this.barHeight(serie),
                events: {
                    selection: function(e) { return display.selectPointsByDrag(this, e); },
                    selectedpoints: function(e) { display.selectedPoints(this, e) },
                    click: function() { display.unselectByClick(this) },
                    load: this.handleLoadChart.bind(this, serie)
                },
                zoomType: 'xy'
            },
            title: { text: '' },
            xAxis: { categories: this._categories(serie), title: { text: null } },
            yAxis: { min: 0, title: { text: this.state.selection.indicator }, labels: { overflow: 'justify' } },
            plotOptions: { bar: { dataLabels: { enabled: true } }, series: {
                states: {
                    select: {
                        color: '#e6db74'
                    }
                },
                point: {
                    selected: true,
                    events: {
                        click: function(e) { handleSelectDrillCategory(e, this, serie); }
                    }
                }
            } },
            legend: { enabled: false },
            credits: { enabled: false },
            series: [ this._serieOf(serie) ]
        };
        return (<HighchartsReact highcharts={Highcharts} options={options} />);
    };

    barHeight = (serie) => {
        const min = 200;
        const length = this._categories(serie).length;
        return (min + (length*20)) + "px";
        //return (650 * ((length > 40 ? Math.round(length / 40) : 0) + 1)) + "px";
    };

    refresh = (info) => {
        this.setState({info});
    };

    refreshSeries = (series) => {
        this.setState({series});
    };

    refreshSelection = (selection) => {
        this.setState({selection});
        Highcharts.charts.filter(chart => chart != null).forEach(chart => this.refreshChart(chart, chart.serie));
    };

    showLoading = () => {
        this.setState({loading:true});
    };

    hideLoading = () => {
        this.setState({loading:false});
    };

    _categories = (serie) => {
        return this.namesOf(this.sort(this.elements(this.state.info.drillCategories, serie)));
    };

    _serieOf = (serie) => {
        return { name: serie.name, color: serie.color, data: this.valuesOf(this.sort(this.elements(this.state.info.drillCategories, serie))), showInLegend: false };
    };

    handleSelectIndicator = (indicator) => {
        this.requester.selectIndicator(indicator.value);
        const selection = this.state.selection;
        this.setState({ selection: { indicator: indicator.value, drill: selection.drill, divideBy: selection.divideBy }});
    };

    handleSelectDrill = (category) => {
        this.requester.selectDrillCategory(category.value);
        const selection = this.state.selection;
        this.setState({ selection: { indicator: selection.indicator, drill: category.value, divideBy: selection.divideBy }});
    };

    handleSelectDivideBy = (category) => {
        this.requester.selectDivideByCategory(category.value);
        const selection = this.state.selection;
        this.setState({ selection: { indicator: selection.indicator, drill: selection.drill, divideBy: category.value }});
    };

    optionsOf = (list) => {
        return list.map(v => this.optionOf(v));
    };

    optionOf = (value) => {
        if (value == null) return null;
        return { value: value, label: this.translate(value) };
    };

    handleSort = (mode) => {
        this.requester.sort(mode);
        const info = this.state.info;
        info.sort = mode;
        this.setState({info:info});
    };

    sort = (elements) => {
        const sortMode = this.state.info.sort;
        if (sortMode === "greater") return this.sortByGreaterValues(elements);
        else if (sortMode === "lower") return this.sortByLowerValues(elements);
        return this.sortByName(elements);
    };

    sortByName = (elements) => {
        return elements;
    };

    sortByGreaterValues = (elements) => {
        elements.sort(function(a, b) { return b.value - a.value; });
        return elements;
    };

    sortByLowerValues = (elements) => {
        elements.sort(function(a, b) { return a.value - b.value; });
        return elements;
    };

    _drillCategories = () => {
        return this.state.info.drillCategories;
    };

    elements = (categories, serie) => {
        const result = [];
        if (serie == null) return result;
        for (let i=0; i<serie.values.length; i++) {
            if (serie.values[i] == 0) continue;
            result.push({name:categories[i], value:serie.values[i]});
        }
        return result;
    };

    namesOf = (elements) => {
        const result = [];
        for (let i=0; i<elements.length; i++) result.push(elements[i].name);
        return result;
    };

    valuesOf = (elements) => {
        const result = [];
        for (let i=0; i<elements.length; i++) result.push(Math.round((elements[i].value + Number.EPSILON) * 100) / 100);
        return result;
    };

    handleLoadChart = (serie, e) => {
        window.setTimeout(() => {
            e.target.serie = serie;
            this.refreshChart(e.target, serie);
            this.registerLoaded(serie);
        }, 1000);
    };

    registerLoaded = (serie) => {
        this.loadedSeries[serie.name] = true;
    };

    refreshChart = (chart, serie) => {
        const theme = Theme.get();
        const display = this;
        if (chart == null || chart.xAxis == null || chart.xAxis.length <= 0) return;
        this.updateSelection(chart, serie);
        chart.xAxis[0].labelGroup.element.childNodes.forEach(function(label) {
            label.style.cursor = "pointer";
            label.style.fill = theme.palette.primary.main;
            label.onclick = function() {
                const pos = display.drillCategoryPosOf(this.textContent);
                chart.series[0].data[pos].select(null, true);
                display.requester.selectDrillCategory(this.textContent);
            }
        });
    };

    updateSelection = (chart, serie) => {
        const selected = this.state.selection.drill;
        const categories = this._categories(serie);
        for (let i=0; i<categories.length; i++) {
            this.updateDrillCategorySelection(chart, categories[i], i);
        }
    };

    updateDrillCategorySelection = (chart, category, pos) => {
        const point = chart.series[0].data[pos];
        if (point == null) return;
        point.select(this.isSelected(category), true);
    };

    drillCategoryPosOf = (category) => {
        const series = this.state.series.length > 0 ? this.state.series[0] : null;
        const categories = series != null ? this._categories(series) : [];
        for (let i=0; i<categories.length; i++) {
            if (categories[i] == category) return i;
        }
        return -1;
    };

    isSelected = (category) => {
        const selected = this.state.selection.drill;
        for (let i=0; i<selected.length; i++) {
            if (selected[i] == category) return true;
        }
        return false;
    };

    handleSelectDrillCategory = (e, chart, serie) => {
        if (!this.loadedSeries[serie.name]) return;
        chart.select(null, true);
        this.requester.selectDrillCategory(e.target.point.category);
    };

    selectPointsByDrag = (chart, e) => {
        chart.series.forEach(series => {
            series.points.forEach(point => {
                if (point.x >= e.xAxis[0].min && point.x <= e.xAxis[0].max &&
                        point.y >= e.yAxis[0].min && point.y <= e.yAxis[0].max) {
                    point.select(true, true);
                }
            });
        });
        Highcharts.fireEvent(chart, 'selectedpoints', { points: chart.getSelectedPoints() });
        return false;
    };

    selectedPoints = (chart, e) => {
        this.requester.selectDrillCategories(e.points.map(p => p.category));
    };

    unselectByClick = (chart) => {
        const points = chart.getSelectedPoints();
        if (points.length == 0) return;
        points.forEach(point => point.select(false));
        this.requester.selectDrillCategories([]);
    };

}

export default withStyles(styles, { withTheme: true })(withSnackbar(CubeBarChart));
DisplayFactory.register("CubeBarChart", withStyles(styles, { withTheme: true })(withSnackbar(CubeBarChart)));