package io.intino.monet.messaging.emails.util;

import io.intino.monet.messaging.Recipient;
import io.intino.monet.messaging.emails.Email;

import javax.mail.Address;
import javax.mail.MessagingException;
import javax.mail.Multipart;
import javax.mail.Session;
import javax.mail.internet.*;
import java.io.File;
import java.io.IOException;
import java.util.List;

import static javax.mail.Message.RecipientType.*;

public class MimeMessageBuilder {

    private Session session;
    private Email email;
    private String senderAddress;
    private String senderName;

    public MimeMessage build() {
        try {
            MimeMessage message = new MimeMessage(session);
            setSenderInfo(message);
            setRecipientInfo(message, email);
            message.setSubject(email.subject());
            message.setContent(createContent(email));
            return message;
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    private static Multipart createContent(Email email) throws IOException, MessagingException {
        Multipart multipart = new MimeMultipart();
        multipart.addBodyPart(createBody(email.body()));
        for(File attachment : email.body().attachments())
            multipart.addBodyPart(createAttachment(attachment));
        return multipart;
    }

    private static MimeBodyPart createBody(Email.Body body) throws MessagingException {
        MimeBodyPart textPart = new MimeBodyPart();
        textPart.setContent(body.text(), body.contentType());
        return textPart;
    }

    private static MimeBodyPart createAttachment(File attachment) throws IOException, MessagingException {
        MimeBodyPart attachmentPart = new MimeBodyPart();
        attachmentPart.attachFile(attachment);
        return attachmentPart;
    }

    private void setRecipientInfo(MimeMessage message, Email email) throws MessagingException {
        message.setRecipients(TO, internetAddresses(List.of(email.recipients().to())));
        if (!email.recipients().cc().isEmpty()) message.setRecipients(CC, internetAddresses(email.recipients().cc()));
        if (!email.recipients().bcc().isEmpty()) message.setRecipients(BCC, internetAddresses(email.recipients().bcc()));
    }

    private Address[] internetAddresses(List<Recipient> recipients) {
        return recipients.stream().map(Recipient::email).map(this::toInternetAddress).toArray(Address[]::new);
    }

    private InternetAddress toInternetAddress(String email) {
        try {
            return new InternetAddress(email);
        } catch (AddressException e) {
            throw new IllegalArgumentException(e);
        }
    }

    private void setSenderInfo(MimeMessage message) throws Exception {
        message.setFrom(new InternetAddress(senderAddress, senderName));
    }

    public MimeMessageBuilder session(Session session) {
        this.session = session;
        return this;
    }

    public MimeMessageBuilder email(Email email) {
        this.email = email;
        return this;
    }

    public MimeMessageBuilder senderAddress(String senderAddress) {
        this.senderAddress = senderAddress;
        return this;
    }

    public MimeMessageBuilder senderName(String senderName) {
        this.senderName = senderName;
        return this;
    }
}
