package io.intino.monet.messaging.emails.store;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import static java.util.Objects.requireNonNull;

/*
* Indicates which emails have been sent to a recipient and when
* */
public class EmailsSent implements EmailStore.Node {

	private transient EmailStore store;
	private final String recipient;
	private final List<EmailSent> emailsAlreadySent;

	EmailsSent(String recipient) {
		this.recipient = recipient;
		this.emailsAlreadySent = new ArrayList<>();
	}

	public boolean put(String signature, LocalDateTime ts) {
		synchronized (this) {
			EmailSent emailSent = get(signature);
			if (emailSent != null) {
				LocalDateTime lastTimeItWasSent = emailSent.ts;
				emailSent.ts = requireNonNull(ts);
				return sameDay(lastTimeItWasSent, ts.toLocalDate());
			}
			emailSent = new EmailSent(signature, ts);
			emailsAlreadySent.add(emailSent);
			return false;
		}
	}

	public void remove(EmailSent emailSent) {
		if (emailSent == null) return;
		synchronized (this) {
			emailsAlreadySent.remove(emailSent);
		}
	}

	public boolean removeIf(Predicate<EmailSent> predicate) {
		if (predicate == null) return false;
		synchronized (this) {
			return emailsAlreadySent.removeIf(predicate);
		}
	}

	public void clear() {
		synchronized (this) {
			emailsAlreadySent.clear();
		}
	}

	public EmailSent get(String signature) {
		if (signature == null) return null;
		synchronized (this) {
			return emailsAlreadySent.stream().filter(e -> e.id.equals(signature)).findFirst().orElse(null);
		}
	}

	public long getTimeSinceSent(String signature, LocalDateTime reference, TimeUnit unit) {
		if (signature == null) return Integer.MIN_VALUE;
		EmailSent emailSent = get(signature);
		if (emailSent == null) return Integer.MIN_VALUE;
		return unit.toChronoUnit().between(emailSent.ts, reference);
	}

	public boolean wasSentToday(String signature) {
		if (signature == null) return false;
		EmailSent emailSent = get(signature);
		if (emailSent == null) return false;
		return sameDay(emailSent.ts, LocalDate.now());
	}

	private boolean sameDay(LocalDateTime ts1, LocalDate ts2) {
		return ts2.equals(ts1.toLocalDate());
	}

	public List<EmailSent> getAll() {
		synchronized (this) {
			return emailsAlreadySent.stream().sorted(Comparator.comparing(EmailSent::ts)).collect(Collectors.toList());
		}
	}

	@Override
	public String id() {
		return recipient;
	}

	@Override
	public void save() {
		synchronized (this) {
			store.save(this);
		}
	}

	EmailsSent datamart(EmailStore store) {
		this.store = store;
		return this;
	}

	@Override
	public String toString() {
		return "EmailsSent{" +
				"recipient='" + recipient + '\'' +
				", emailsAlreadySent=" + emailsAlreadySent +
				'}';
	}

	public static class EmailSent {

		private final String id;
		private LocalDateTime ts;

		public EmailSent(String id, LocalDateTime ts) {
			this.id = requireNonNull(id);
			this.ts = requireNonNull(ts);
		}

		public String id() {
			return id;
		}

		public LocalDateTime ts() {
			return ts;
		}

		@Override
		public boolean equals(Object o) {
			if (this == o) return true;
			if (o == null || getClass() != o.getClass()) return false;
			EmailSent emailSent = (EmailSent) o;
			return Objects.equals(id, emailSent.id) && Objects.equals(ts, emailSent.ts);
		}

		@Override
		public int hashCode() {
			return Objects.hash(id, ts);
		}

		@Override
		public String toString() {
			return "EmailSent{" +
					"id='" + id + '\'' +
					", ts=" + ts +
					'}';
		}
	}
}
