package io.intino.monet.messaging.emails.store;

import com.google.gson.*;
import io.intino.alexandria.Scale;
import io.intino.alexandria.Timetag;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import static java.nio.file.StandardOpenOption.CREATE;
import static java.nio.file.StandardOpenOption.TRUNCATE_EXISTING;
import static java.util.Objects.requireNonNull;

/**
 * Fully thread-safe json store
 */
public class EmailStore {

	private final File root;
	private final Gson gson;
	private final Map<String, EmailBlacklist> emailsBlacklistMap;
	private final Map<String, EmailsSent> emailsSentMap;
	private volatile boolean readOnly = false;

	public EmailStore(File root) {
		this.root = requireNonNull(root);
		if (root.exists() && !root.isDirectory()) throw new IllegalArgumentException(root + " is not a directory");
		root.mkdirs();
		this.gson = createGson();
		this.emailsBlacklistMap = new HashMap<>();
		this.emailsSentMap = new HashMap<>();
	}

	public boolean readOnly() {
		return readOnly;
	}

	public EmailStore readOnly(boolean readOnly) {
		this.readOnly = readOnly;
		return this;
	}

	public EmailBlacklist emailBlacklist(String recipient) {
		synchronized (this) {
			EmailBlacklist node = emailsBlacklistMap.get(recipient);
			if (node == null) node = load(recipient, EmailBlacklist.class);
			node = node == null ? new EmailBlacklist(recipient) : node;
			emailsBlacklistMap.put(recipient, node);
			return node.datamart(this);
		}
	}

	public List<EmailBlacklist> emailBlacklists(Predicate<EmailBlacklist> predicate) {
		synchronized (this) {
			return emailsBlacklistMap.values().stream().filter(predicate).collect(Collectors.toList());
		}
	}

	public EmailsSent emailsSent(String recipient) {
		synchronized (this) {
			EmailsSent node = emailsSentMap.get(recipient);
			if (node == null) node = load(recipient, EmailsSent.class);
			node = node == null ? new EmailsSent(recipient) : node;
			emailsSentMap.put(recipient, node);
			return node.datamart(this);
		}
	}

	public List<EmailsSent> emailsSent(Predicate<EmailsSent> predicate) {
		synchronized (this) {
			return emailsSentMap.values().stream().filter(predicate).collect(Collectors.toList());
		}
	}

	public void save() {
		if (readOnly) return;
		synchronized (this) {
			emailsBlacklistMap.values().forEach(Node::save);
			emailsSentMap.values().forEach(Node::save);
		}
	}

	void save(Node node) {
		if (readOnly) return;
		if (node == null) return;
		synchronized (this) {
			serialize(node);
		}
	}

	private void serialize(Node node) {
		if (readOnly) return;
		synchronized (this) {
			try {
				File file = new File(root(), filename(node));
				file.getParentFile().mkdirs();
				Files.writeString(file.toPath(), toJson(node), CREATE, TRUNCATE_EXISTING);
			} catch (IOException e) {
				throw new EmailStoreException(e.getMessage(), e);
			}
		}
	}

	private <T extends Node> T load(String id, Class<T> type) {
		return loadFile(filename(type, id), type);
	}

	private <T extends Node> T loadFile(String filename, Class<T> type) {
		synchronized (this) {
			File file = new File(root(), filename);
			String json = read(file);
			try {
				return json == null ? null : gson.fromJson(json, type);
			} catch (Exception e) {
				throw new EmailStoreException(e.getMessage(), e);
			}
		}
	}

	private String read(File file) {
		try {
			if (!file.exists()) return null;
			return Files.readString(file.toPath());
		} catch (IOException e) {
			throw new EmailStoreException(e.getMessage(), e);
		}
	}

	public File root() {
		return root;
	}

	private String filename(Node node) {
		return filename(node.getClass(), node.id());
	}

	private String filename(Class<?> clazz, String id) {
		return clazz.getSimpleName() + "/" + id + ".json";
	}

	public String toJson(Node node) {
		synchronized (this) {
			return gson.toJson(node);
		}
	}

	private Gson createGson() {
		GsonBuilder builder = new GsonBuilder();
		builder.setPrettyPrinting();

		builder.registerTypeAdapter(LocalDateTime.class, (JsonSerializer<LocalDateTime>) (ts, type, jsonSerializationContext)
				-> new JsonPrimitive(Timetag.of(ts, Scale.Minute).value()));

		builder.registerTypeAdapter(LocalDateTime.class, (JsonDeserializer<LocalDateTime>) (jsonElement, type, jsonSerializationContext)
				-> Timetag.of(jsonElement.getAsString()).datetime());

		return builder.create();
	}

	public interface Node {
		String id();

		void save();
	}
}
