package io.intino.monet.messaging.emails;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;

import static java.util.Objects.requireNonNull;

// The signatures are hierarchical
// EmailSignature = EmailType[.subtype1.subtype2.subtype3...]
public class EmailSignature {

	public static final String SEPARATOR = "\\.";

	public static EmailSignature wrap(String signature) {
		int sepIndex = signature.indexOf(SEPARATOR);
		if(sepIndex < 0) return new EmailSignature(signature);
		String type = signature.substring(0, sepIndex);
		String[] subtypes = signature.substring(sepIndex + 1).split(SEPARATOR);
		return new EmailSignature(type, subtypes);
	}

	private String type;
	private List<String> subTypes;
	private String signature;

	public EmailSignature(String type, String... subTypes) {
		init(type, subTypes);
	}

	public String get() {
		return signature;
	}

	private void init(String type, String... subTypes) {
		this.type = requireNonNull(type);
		checkSubTypes(subTypes);
		this.subTypes = empty(subTypes) ? Collections.emptyList() : Arrays.asList(subTypes);
		this.signature = empty(subTypes) ? type : type + SEPARATOR + String.join(SEPARATOR, subTypes);
	}

	private void checkSubTypes(String[] subTypes) {
		if(Arrays.stream(subTypes).anyMatch(s -> s == null || s.isBlank()))
			throw new IllegalArgumentException("Subtypes cannot be null nor blank");
	}

	private boolean empty(String[] subTypes) {
		return subTypes == null || subTypes.length == 0;
	}

	public String type() {
		return type;
	}

	public boolean hasSubtype(String subType) {
		return subTypes.stream().anyMatch(s -> s.equals(subType));
	}

	public String subType(int index) {
		return index < subTypes.size() ? subTypes.get(index) : null;
	}

	public List<String> subTypes() {
		return Collections.unmodifiableList(subTypes);
	}

	// A, A -> true
	// A, B -> false
	// B, A -> false,
	// A, A.B -> true
	// B, A.B -> false
	// A.B, A -> false
	// A.B, B -> false
	public boolean contains(EmailSignature other) {
		return other.signature.contains(signature);
	}

	@Override
	public boolean equals(Object o) {
		if (this == o) return true;
		if (o == null || getClass() != o.getClass()) return false;
		EmailSignature that = (EmailSignature) o;
		return Objects.equals(signature, that.signature);
	}

	@Override
	public int hashCode() {
		return Objects.hash(signature);
	}

	@Override
	public String toString() {
		return signature;
	}
}
