package io.intino.monet.messaging.pushnotifications;

import io.intino.alexandria.Json;
import io.intino.alexandria.logger.Logger;
import io.intino.monet.messaging.emails.util.JsonPretty;
import io.intino.monet.messaging.logging.MessagingLog;

import java.io.File;
import java.io.IOException;
import java.io.Serializable;
import java.nio.file.Files;
import java.util.ArrayList;
import java.util.List;

import static java.nio.file.StandardCopyOption.REPLACE_EXISTING;
import static java.nio.file.StandardOpenOption.CREATE;
import static java.util.Objects.requireNonNull;

public class PushNotificationStore {

    private final File root;
    private final int maxNotificationsPerUser;
    private final Object mutex = new Object();

    public PushNotificationStore(File root) {
        this(root, 64);
    }

    public PushNotificationStore(File root, int maxNotificationsPerUser) {
        this.root = requireNonNull(root);
        root.mkdirs();
        this.maxNotificationsPerUser = maxNotificationsPerUser;
    }

    public List<PushNotification> loadNotifications(String userEmail) {
        synchronized (mutex) {
            File file = new File(root, userEmail + ".json");
            if(!file.exists() || file.length() == 0) return new ArrayList<>();
            try {
                String json = Files.readString(file.toPath());
                NotificationList notificationList = Json.fromString(json, NotificationList.class);
                return notificationList.get();
            } catch (IOException e) {
                MessagingLog.error("Failed to read notifications datamart " + file.getName() + ": " + e.getMessage(), e);
                try {Files.copy(file.toPath(), new File(file.getAbsolutePath() + ".error").toPath(), REPLACE_EXISTING);} catch (IOException ignored) {}
            } catch (Exception e) {
                MessagingLog.error("Failed to read notifications datamart " + file.getName() + ": " + e.getMessage(), e);
            }
        }
        return new ArrayList<>();
    }

    public void saveNotifications(String userEmail, List<PushNotification> notifications) {
        synchronized (mutex) {
            File file = new File(root, userEmail + ".json");
            try {
                File tmp = new File(file.getAbsolutePath() + ".tmp");
                Files.writeString(tmp.toPath(), JsonPretty.toJson(new NotificationList(notifications)), CREATE);
                Files.move(tmp.toPath(), file.toPath(), REPLACE_EXISTING);
            } catch (Exception e) {
                MessagingLog.error("Failed to write notifications datamart " + file.getName() + ": " + e.getMessage(), e);
            }
        }
    }

    public void addNotification(PushNotification notification) {
        synchronized (mutex) {
            String recipient = notification.recipient().email();
            List<PushNotification> notifications = loadNotifications(recipient);
            notifications.add(notification);
            saveNotifications(recipient, notifications);
        }
    }

    private class NotificationList implements Serializable {

        public List<PushNotification> notifications;

        public NotificationList(List<PushNotification> notifications) {
            this.notifications = new ArrayList<>(notifications);
            this.notifications.sort(this::compareNotifications);
            this.notifications = new ArrayList<>(this.notifications.subList(Math.max(notifications.size() - maxNotificationsPerUser, 0), notifications.size()));
        }

        public List<PushNotification> get() {
            return notifications == null ? new ArrayList<>() : notifications;
        }

        private int compareNotifications(PushNotification n1, PushNotification n2) {
            if(n1.ts() == null && n2.ts() == null) return 0;
            if(n1.ts() != null) return -1;
            if(n2.ts() != null) return 1;
            return n1.ts().compareTo(n2.ts());
        }
    }
}
