package io.intino.monet.messaging.logging;

import io.intino.alexandria.logger.Logger;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicReference;

import static io.intino.alexandria.logger.Logger.Level.*;

public class MessagingLog {

	private static final List<LogHandler> Handlers = new ArrayList<>(1);
	public static final AtomicReference<Logger.Level> MaxLevel = new AtomicReference<>(INFO);

	public static void addHandler(LogHandler handler) {
		if (handler == null) return;
		Handlers.add(handler);
	}

	public static void close() {
		Handlers.forEach(LogHandler::close);
	}

	private static void publish(Logger.Level level, String message) {
		try {
			if (level.ordinal() > MaxLevel.get().ordinal()) return;
			Handlers.forEach(handler -> handler.publish(level, message));
		} catch (Throwable ignored) {
		}
	}

	public static void trace(String message) {
		publish(TRACE, format(TRACE, message));
	}

	public static void debug(String message) {
		publish(DEBUG, format(DEBUG, message));
	}

	public static void info(String message) {
		publish(INFO, format(INFO, message));
	}

	public static void notification(String message) {
		publish(NOTIFICATION, format(NOTIFICATION, message));
	}

	public static void warn(String message) {
		publish(WARN, format(WARN, message));
	}

	public static void error(String message) {
		publish(ERROR, format(ERROR, message));
	}

	public static void error(Throwable e) {
		publish(ERROR, format(e));
	}

	public static void error(String message, Throwable e) {
		publish(ERROR, format(message, e));
	}

	private static String format(String message, Throwable e) {
		return "[%level]\nts: %date\nsource: %C\nmessage: %m\n".replace("%level", ERROR.name()).replace("%date", Instant.now().toString()).replace("%C", caller()).replace("%m", formatMessage(message, e));
	}

	private static String format(Throwable e) {
		return "[%level]\nts: %date\nsource: %C\nmessage: %m\n".replace("%level", ERROR.name()).replace("%date", Instant.now().toString()).replace("%C", caller()).replace("%m", formatMessage(e));
	}

	private static String format(Logger.Level level, String message) {
		return "[%level]\nts: %date\nsource: %C\nmessage: %m\n".replace("%level", level.name()).replace("%date", Instant.now().toString()).replace("%C", caller()).replace("%m", formatMessage(message)) + "\n";
	}

	private static String formatMessage(String message, Throwable e) {
		if (message == null) {
			return formatMessage(e);
		} else {
			String var10000 = formatMessage(message);
			return "\n\t" + var10000 + (e != null ? "\n\t\n\tCaused by:" + formatMessage(e) : "");
		}
	}

	private static String formatMessage(String message) {
		return !message.contains("\n") ? message : "\n\t" + message.replace("\n", "\n\t");
	}

	private static String formatMessage(Throwable e) {
		if (e == null) {
			return "Null";
		} else {
			StringWriter writer = new StringWriter();
			e.printStackTrace(new PrintWriter(writer));
			String message = writer.toString();
			return "\n\t" + message.replace("\n", "\n\t");
		}
	}

	private static String caller() {
		StackTraceElement[] stElements = Thread.currentThread().getStackTrace();

		for (int i = 4; i < stElements.length; ++i) {
			StackTraceElement ste = stElements[i];
			if (!ste.getClassName().equals(Logger.class.getName()) && !ste.getClassName().startsWith("java.util.ArrayList") && !ste.getClassName().startsWith("java.lang.Thread")) {
				String clazz = ste.getClassName();
				return clazz + ":" + ste.getMethodName() + ":" + ste.getLineNumber();
			}
		}

		return "Unknown";
	}
}
