package io.intino.monet.messaging.emails.store;

import io.intino.monet.messaging.emails.EmailSignature;
import io.intino.monet.messaging.emails.util.JsonPretty;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import static java.util.Objects.requireNonNull;

/*
* Indicates what types of emails a recipient should NOT receive
* */
public class EmailBlacklist implements EmailStore.Node {

	private transient EmailStore store;
	private final String recipient;
	private final List<String> blacklist;

	EmailBlacklist(String recipient) {
		this.recipient = requireNonNull(recipient);
		this.blacklist = new ArrayList<>();
	}

	public String recipient() {
		return recipient;
	}

	public Collection<String> getAll() {
		synchronized (this) {
			return List.copyOf(blacklist);
		}
	}

	public boolean add(EmailSignature signature) {
		return add(signature.get());
	}

	public boolean add(String signature) {
		synchronized (this) {
			if (blacklist.contains(signature)) return false;
			return blacklist.add(signature);
		}
	}

	public void remove(EmailSignature signature) {
		remove(signature.toString());
	}

	public void remove(String signature) {
		synchronized (this) {
			blacklist.remove(signature);
		}
	}

	public void clear() {
		synchronized (this) {
			blacklist.clear();
		}
	}

	public boolean contains(String signature) {
		return contains(new EmailSignature(signature));
	}

	public boolean contains(EmailSignature signature) {
		synchronized (this) {
			return blacklist.stream().anyMatch(s -> new EmailSignature(s).contains(signature));
		}
	}

	@Override
	public String toString() {
		return JsonPretty.toJson(this);
	}

	@Override
	public String id() {
		return recipient;
	}

	@Override
	public void save() {
		synchronized (this) {
			store.save(this);
		}
	}

	EmailBlacklist datamart(EmailStore store) {
		this.store = store;
		return this;
	}
}
