package io.intino.monet.messaging;

import io.intino.alexandria.logger.Logger;
import io.intino.monet.messaging.emails.EmailService;
import io.intino.monet.messaging.emails.store.EmailStore;
import io.intino.monet.messaging.pushnotifications.PushNotificationService;
import io.intino.monet.messaging.pushnotifications.PushNotificationStore;

import java.io.File;

import static java.util.Objects.requireNonNull;

public class Messaging {

    private final File templatesRoot;
    private final RecipientsStore recipientsStore;
    private final Emails emails;
    private final PushNotifications pushNotifications;

    public Messaging(File templatesRoot, RecipientsStore recipientsStore,
                     EmailService emailService, PushNotificationService notificationService) {
        this.templatesRoot = requireNonNull(templatesRoot);
        this.recipientsStore = recipientsStore;
        this.emails = new Emails(requireNonNull(emailService));
        this.pushNotifications = new PushNotifications(requireNonNull(notificationService));
    }

    public File templatesRoot() {
        return templatesRoot;
    }

    public RecipientsStore recipientsStore() {
        return recipientsStore;
    }

    public Emails emails() {
        return emails;
    }

    public PushNotifications pushNotifications() {
        return pushNotifications;
    }

    public class Emails {

        private final EmailService emailService;
        private final EmailStore store;
        private final MessageBlacklist blacklist;

        private Emails(EmailService emailService) {
            this.emailService = emailService;
            this.store = emailService.store();
            this.blacklist = new MessageBlacklist();
        }

        public EmailService service() {
            return emailService;
        }

        public EmailStore store() {
            return store;
        }

        public MessageBlacklist blacklist() {
            return blacklist;
        }

        public NotificationTemplate getEmailTemplate(String type, String language) {
            return NotificationTemplate.load(new File(templatesRoot(), "email"), type, language);
        }
    }

    public class PushNotifications {

        private final PushNotificationService pushNotificationService;
        private final PushNotificationStore store;
        private final MessageBlacklist blacklist;

        PushNotifications(PushNotificationService pushNotificationService) {
            this.pushNotificationService = pushNotificationService;
            this.store = pushNotificationService.store();
            this.blacklist = new MessageBlacklist();
        }

        public PushNotificationService service() {
            return pushNotificationService;
        }

        public MessageBlacklist blacklist() {
            return blacklist;
        }

        public PushNotificationStore store() {
            return store;
        }

        public NotificationTemplate getPushTemplate(String type, String language) {
            return NotificationTemplate.load(new File(templatesRoot(), "push"), type, language);
        }
    }

    public static class Builder {

        private File templatesRoot;
        private RecipientsStore recipientsStore;
        private EmailService emailService;
        private PushNotificationService notificationService;

        public Messaging build() {
            return new Messaging(templatesRoot, recipientsStore, emailService, notificationService);
        }

        public Builder templatesRoot(File templatesRoot) {
            this.templatesRoot = templatesRoot;
            return this;
        }

        public Builder recipientsStore(RecipientsStore recipientsStore) {
            this.recipientsStore = recipientsStore;
            return this;
        }

        public Builder emailService(EmailService emailService) {
            this.emailService = emailService;
            return this;
        }

        public Builder notificationService(PushNotificationService notificationService) {
            this.notificationService = notificationService;
            return this;
        }
    }
}
