package io.intino.monet.engine.edition.editors;

import io.intino.alexandria.logger.Logger;
import io.intino.monet.engine.edition.Edition;
import io.intino.monet.engine.edition.FormStore;
import io.intino.monet.engine.edition.Language;
import io.intino.monet.engine.edition.editors.util.DynamicSourceHelper;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.util.*;
import java.util.stream.Collectors;

import static java.util.Collections.emptyList;

public class OptionMultipleEdition extends Edition {

	private final File thesaurusRepository;
	private final String optionsSource;
	private final Set<String> dependentFields;
	protected List<String> options;

	public OptionMultipleEdition(String name, FormStore store, Map<String, String> options, Language language) {
		super(name, store, options, language);
		this.thesaurusRepository = new File(options.get("thesaurus-repository"));
		this.optionsSource = options.get("values." + language);
		this.dependentFields = DynamicSourceHelper.getVariables(optionsSource);
	}

	@Override
	protected void onModified(String name) {
		if(isStaticSource()) return;
		if(dependentFields().contains(name))
			set("");
	}

	@Override
	public Edition init() {
		super.init();
		if(isStaticSource() && options == null) options = loadOptions();
		return this;
	}

	private List<String> loadOptions() {
		if (isStaticSource()) return toList(optionsSource);
		String filename = sourceFilename();
		return filename != null ? optionsFromFile(filename) : emptyList();
	}

	private boolean isStaticSource() {
		return !optionsSource.startsWith("@");
	}

	private List<String> optionsFromFile(String filename) {
		try {
			File thesaurus = new File(thesaurusRepository + "/" + filename);
			return thesaurus.exists() ? Files.readAllLines(thesaurus.toPath()) : emptyList();
		} catch (IOException e) {
			Logger.error(e);
			return emptyList();
		}
	}

	private String sourceFilename() {
		return DynamicSourceHelper.getDynamicRoute(optionsSource).map(this::storeValueOf).collect(Collectors.joining(""));
	}

	private Set<String> dependentFields() {
		return dependentFields;
	}

	public List<String> options() {
		return options == null ? loadOptions() : options;
	}

	public String[] get() {
		String value = (String) value();
		return value == null || value.isEmpty() ? new String[0] : value.split(";");
	}

	public void set(String... values) {
		this.store.put(name, String.join(";",values));
	}

	private List<String> toList(String values) {
		if (values == null) return emptyList();
		return Arrays.stream(values.split(";"))
				.map(String::trim)
				.collect(Collectors.toList());
	}

}
