package io.intino.monet.engine.edition;

import io.intino.alexandria.logger.Logger;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.time.LocalDate;
import java.util.*;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import static java.nio.file.StandardOpenOption.APPEND;
import static java.nio.file.StandardOpenOption.CREATE;
import static java.util.Objects.requireNonNull;

@SuppressWarnings("ResultOfMethodCallIgnored")
public class LocalFormStore implements FormStore {
	private final File folder;
	private final Map<String, Object> data;
	private final List<Consumer<String>> onModifiedListeners = new ArrayList<>();

	public LocalFormStore(File folder) {
		this.folder = folder;
		this.folder.mkdirs();
		this.data = new HashMap<>();
	}

	public File folder() {
		return folder;
	}

	public void load() throws IOException {
		this.data.clear();
		this.data.putAll(read(target().toPath()));
	}

	public void save() throws IOException {
		target().delete();
		write(target().toPath());
	}

	private Map<String, Object> read(Path path) throws IOException {
		List<String> lines = Files.readAllLines(path);
		return lines.stream().map(this::parse).filter(Objects::nonNull).collect(Collectors.toMap(v->v.key, v->v.value, (v,m) -> m));
	}

	private void write(Path path) throws IOException {
		for (String key : data.keySet())
			Files.write(path, new Variable(key, data.get(key)).serialize(), CREATE, APPEND);
	}

	private Variable parse(String line) {
		return !line.isEmpty() ? new Variable(line.split("\t")) : null;
	}

	@Override
	public List<String> keys() {
		return new ArrayList<>(data.keySet());
	}

	@Override
	public Object get(String name) {
		Object value = data.get(name);
		return value != null ? get(value, file(name)) : null;
	}

	private Object get(Object value, File file) {
		return file != null ? new Object[] {value, file} : value;
	}

	private File file(String name) {
		File[] files = folder.listFiles(f -> withoutExtension(f.getName()).equals(name));
		return files == null || files.length == 0 ? null : files[0];
	}

	private String withoutExtension(String name) {
		int extensionBegin = name.lastIndexOf('.');
		return extensionBegin < 0 ? name : name.substring(0, extensionBegin);
	}

	@Override
	public void put(String name, Object value) {
		if(update(name, value))
			onModifiedListeners.forEach(listener -> listener.accept(name));
	}

	private boolean update(String name, Object value) {
		if (value != null) return add(name, value);
		else return remove(name);
	}

	private boolean add(String name, Object value) {
		if (value instanceof File) return move(name, (File) value);
		else if (value instanceof InputStream) return save(name, (InputStream) value);
		else if(value instanceof Object[]) return Arrays.stream((Object[]) value).map(v -> add(name, v)).reduce((a, b) -> a | b).orElse(false);
		else {
			Object oldValue = data.put(name, value);
			return !Objects.equals(value, oldValue);
		}
	}

	private boolean remove(String name) {
		return data.remove(name) != null && findAndDeleteFile(name);
	}

	private boolean findAndDeleteFile(String name) {
		File[] files = folder.listFiles(f -> f.isFile() && f.getName().startsWith(name + "."));
		if(files == null || files.length == 0) return false;
		return files[0].delete();
	}

	private boolean move(String name, File file) {
		findAndDeleteFile(name);
		File dest = new File(folder, name + getExtension(file));
		dest.getParentFile().mkdirs();
		file.renameTo(dest);
		return true;
	}

	private String getExtension(File file) {
		String name = file.getName();
		int extensionBegin = name.lastIndexOf('.');
		return extensionBegin < 0 ? "" : name.substring(extensionBegin);
	}

	private boolean save(String name, InputStream is) {
		try {
			File destination = new File(folder, name);
			destination.getParentFile().mkdirs();
			Files.write(destination.toPath(), is.readAllBytes(), StandardOpenOption.CREATE);
			return true;
		} catch (IOException e) {
			Logger.error(e);
			return false;
		}
	}

	@Override
	public int size() {
		return data.size();
	}

	@Override
	public void onModified(Consumer<String> listener) {
		onModifiedListeners.add(requireNonNull(listener));
	}

	private File target() {
		return new File(folder, "data.tsv");
	}

	private static class Variable {
		public String key;
		public Object value;

		public Variable(String[] split) {
			this.key = split[0];
			this.value = parse(split.length > 1 ? split[1] : "", split.length > 2 ? split[2] : "");
		}

		private Object parse(String type, String value) {
			if (type.equalsIgnoreCase("Double")) return Double.parseDouble(value);
			if (type.equalsIgnoreCase("LocalDate")) return LocalDate.parse(value);
			return value;
		}

		public Variable(String key, Object value) {
			this.key = key;
			this.value = value;
		}

		public byte[] serialize() {
			return (key + '\t' + typeOf(value) + '\t' + value + '\n').getBytes();
		}

		private String typeOf(Object value) {
			return value.getClass().getSimpleName();
		}
	}
}
