package io.intino.monet.engine;

import io.intino.alexandria.logger.Logger;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.util.*;
import java.util.function.BiConsumer;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static java.util.Comparator.comparing;
import static java.util.stream.Collectors.*;

public class OrderTypes {

	private static final Supplier<Map<String, Record>> MapFactory = HashMap::new;;
	public static final String TEMPLATE_EXTENSION = ".docx";
	private static volatile Map<String, Record> records = MapFactory.get();

	private static final Map<String, BiConsumer<Record, String>> setters = new HashMap<>() {{
		put("label.en", (r, v) -> r.labelEn = v);
		put("label.es", (r, v) -> r.labelEs = v);
		put("label.pt", (r, v) -> r.labelPt = v);
		put("hint.en", (r, v) -> r.hintEn = v);
		put("hint.es", (r, v) -> r.hintEs = v);
		put("hint.pt", (r, v) -> r.hintPt = v);
		put("code", (r, v) -> r.code = v);
		put("effort", (r, v) -> r.effort = Integer.parseInt(v));
		put("input", (r, v) -> r.input = v);
		put("calculations", (r, v) -> r.calculations = v);
		put("annexes", (r, v) -> r.annexes = v);
		put("target", (r, v) -> r.target = v);
		put("channel", (r, v) -> r.channel = Channel.valueOf(v));
		put("parent", (r, v) -> r.parent = v);
		put("assertion", (r, v) -> {
			String[] split = v.split(":");
			r.assertionCode = split[0];
			r.assertionAttr = split.length == 1 ? Collections.emptyMap() :
					Arrays.stream(split[1].split(",")).map(s -> s.split(" as "))
							.collect(toMap(s -> s[0].trim(), s -> s.length == 1 ? s[0].trim() : s[1].trim(), (v1, v2) -> v1, LinkedHashMap::new));
		});
		put("report.filename", (r, v) -> r.reportFilename = v);
		put("visible", (r, v) -> r.visible = "true".equalsIgnoreCase(v));
	}};

	public static void init(File file) {
		try {
			Map<String, Record> newRecords = MapFactory.get();

			Files.readAllLines(file.toPath()).stream()
					.filter(l -> !l.trim().isEmpty())
					.map(l -> l.split("\t", -1))
					.forEach(l -> setters.getOrDefault(l[1], nullSetter()).accept(record(newRecords, l[0]), l[2]));

			newRecords.values().stream().sorted(recordsWithNoParentsFirst()).forEach(r -> load(file, r, newRecords));

			records = newRecords;

		} catch (IOException e) {
			Logger.error(e);
		}
	}

	private static Comparator<Record> recordsWithNoParentsFirst() {
		return (r1, r2) -> {
			if(r1.parent == null && r2.parent == null) return 0;
			return r1.parent == null ? -1 : 1;
		};
	}

	private static void load(File file, Record record, Map<String, Record> records) {
		if(record.parent != null) {
			loadParentInfo(file, record, records);
			return;
		}
		loadChecklistAndTriples(file, record, record.code);
	}

	private static void loadChecklistAndTriples(File file, Record record, String name) {
		File checklistFile = checklistFile(file.getParentFile(), name);
		Map<String, List<String[]>> triples = Checklist.triples(checklistFile);
		record.checklist = Checklist.load(checklistFile);
		setTriples(record, triples);
	}

	private static void loadParentInfo(File file, Record record, Map<String, Record> records) {
		if(record.parent == null) return;
		if(record.parent.equals(record.code)) throw new IllegalStateException("Order-Type parent cannot be itself (" + record.code + ")");

		Record parent = records.get(record.parent);
		if(parent == null) return;

		if(parent.parent != null && parent.parent.equals(record.code)) throw new IllegalStateException("Cyclic inheritance is not allowed (" + record.code + "-" + parent.code + ")");

		inheritValuesFromParent(record, parent);

		loadChecklistFromParent(file, record, records);
	}

	private static void loadChecklistFromParent(File file, Record record, Map<String, Record> records) {

		Checklist parentChecklist = getChecklist(file.getParentFile(), record.parent);
		Checklist childChecklist = getChecklist(file.getParentFile(), record.code);

		// If child has no checklist, just load the parent's one
		if(childChecklist == null) {
			loadChecklistAndTriples(file, record, record.parent);
			return;
		}

		record.checklist = parentChecklist == null ? childChecklist : Checklist.merge(parentChecklist, childChecklist);

		Map<String, List<String[]>> triples = parentChecklist != null ? Checklist.triples(checklistFile(file.getParentFile(), record.parent)) : new HashMap<>();
		triples.putAll(Checklist.triples(checklistFile(file.getParentFile(), record.code)));
		setTriples(record, triples);
	}

	private static void setTriples(Record record, Map<String, List<String[]>> triples) {
		record.triples = record.checklist.fields.stream()
				.sorted(Checklist.fieldCodeComparator())
				.collect(toMap(e -> e, e -> Checklist.asMap(Checklist.triples(
						triples.getOrDefault(e.code, Collections.emptyList()))), (a, v) -> a, LinkedHashMap::new));
	}

	private static Checklist getChecklist(File root, String name) {
		File file = checklistFile(root, name);
		return file.exists() ? Checklist.load(file) : null;
	}

	private static File checklistFile(File root, String name) {
		return new File(root, "order-types/" + name + ".triples");
	}

	private static void inheritValuesFromParent(Record record, Record parent) {
		if(record.labelEn == null) record.labelEn = parent.labelEn;
		if(record.labelEs == null) record.labelEs = parent.labelEs;
		if(record.labelPt == null) record.labelPt = parent.labelPt;
		if(record.hintEn == null) record.hintEn = parent.hintEn;
		if(record.hintEs == null) record.hintEs = parent.hintEs;
		if(record.hintPt == null) record.hintPt = parent.hintPt;
		if(record.target == null) record.target = parent.target;
		if(record.effort == 0) record.effort = parent.effort;
		if(record.input == null) record.input = parent.input;
		if(record.calculations == null) record.calculations = parent.calculations;
		if(record.annexes == null) {record.annexes = parent.annexes; record.codeForAnnexes = parent.codeForAnnexes();}
		if(record.channel == null) record.channel = parent.channel;
		if(record.assertionCode == null) record.assertionCode = parent.assertionCode;
		if(record.assertionAttr == null) record.assertionAttr = parent.assertionAttr;
		if(record.reportFilename == null) record.reportFilename = parent.reportFilename;
		if(record.visible == null) record.visible = parent.visible;
	}

	private static BiConsumer<Record, String> nullSetter() {
		return (record, s) -> {};
	}

	public static Collection<Record> all() {
		return records.values();
	}

	public static Record of(String code) {
		return records.get(code);
	}

	private static Record record(Map<String, Record> records, String id) {
		if (!records.containsKey(id)) {
			Record record = new Record();
			record.code = id;
			records.put(id, record);
		}
		return records.get(id);
	}

	public enum Channel {web, app, both}

	public static class Record {
		public static final String ANNEX_SEPARATOR = "$";
		private String code;
		private String labelEn;
		private String labelEs;
		private String labelPt;
		private String hintEn;
		private String hintEs;
		private String hintPt;
		private String target;
		private int effort;
		private String input;
		private String calculations;
		private String annexes;
		private Channel channel;
		private String assertionCode;
		private String parent;
		private Checklist checklist = new Checklist();
		private Map<Checklist.Field, Map<String, String>> triples = new HashMap<>();
		private Map<String, String> assertionAttr;
		private String reportFilename;
		private String codeForAnnexes;
		private Boolean visible;

		public String code() {
			return code;
		}

		public String label(String lang) {
			return lang.equals("en") ? labelEn : lang.equals("es") ? labelEs : labelPt;
		}

		public String hint(String lang) {
			return lang.equals("en") ? hintEn : lang.equals("es") ? hintEs : hintPt;
		}

		public String category() {
			return code.startsWith("P") ? "Preventive" : code.startsWith("A") ? "Administrative" : "Corrective";
		}

		public int effort() {
			return effort;
		}

		public boolean isManual() {
			return input == null || input.isEmpty();
		}

		public List<String> input() {
			if (input == null) return Collections.emptyList();
			return Stream.of(input.split(",")).map(String::trim).filter(s -> !s.isEmpty()).collect(toList());
		}

		public List<String> calculations() {
			if (calculations == null) return Collections.emptyList();
			return Stream.of(calculations.split(",")).map(String::trim).filter(s -> !s.isEmpty()).collect(toList());
		}

		public List<String> annexes() {
			if (annexes == null) return Collections.emptyList();
			return Stream.of(annexes.split(",")).map(String::trim).filter(s -> !s.isEmpty()).collect(toList());
		}

		public String codeForAnnexes() {
			return codeForAnnexes != null ? codeForAnnexes : code;
		}

		public String annexFilenameOf(String annexName) {
			String filename = annexes().stream().filter(a -> a.startsWith(annexName))
					.map(a -> a.substring(annexName.length() + 1))
					.findFirst().orElse(null);
			return codeForAnnexes() + ANNEX_SEPARATOR + filename + TEMPLATE_EXTENSION;
		}

		public String templateFilename() {
			return code + TEMPLATE_EXTENSION;
		}

		public String target() {
			return target;
		}

		public Channel channel() {
			return channel;
		}

		public String assertionCode() {
			return assertionCode;
		}

		public Map<String, String> assertionAttrs() {
			return assertionAttr;
		}

		public String reportFilename() {
			return reportFilename;
		}

		public Checklist checklist() {
			return checklist;
		}

		public Map<Checklist.Field, Map<String, String>> triples() {
			return triples;
		}

		public String parent() {
			return parent;
		}

		public boolean visible() {
			return visible == null || visible;
		}
	}

	public static class Checklist implements Iterable<Checklist.Field> {

		private final List<Field> fields = new ArrayList<>();

		private static Checklist load(File file) {
			return load(triples(file));
		}

		private static Checklist load(Map<String, List<String[]>> triples) {
			Checklist checklist = new Checklist();
			for (String key : triples.keySet())
				checklist.add(key, triples(triples.get(key)));
			init(checklist);
			return checklist;
		}

		private static Checklist merge(Checklist parent, Checklist child) {
			Checklist checklist = new Checklist();
			checklist.fields.addAll(parent.fields);

			for(Field f : child) {
				removeIfAlreadyExists(checklist, f);
				append(checklist, f);
			}

			init(checklist);

			return checklist;
		}

		private static void removeIfAlreadyExists(Checklist checklist, Field f) {
			checklist.fields.removeIf(field -> field.name.equals(f.name));
		}

		private static void append(Checklist checklist, Field f) {
			checklist.fields.add(f);
		}

		private static void init(Checklist checklist) {
			sort(checklist.fields);
			checklist.setMarkers();
		}

		private static void sort(List<Field> fields) {
			fields.sort(fieldCodeComparator());
		}

		private static Comparator<Field> fieldCodeComparator() {
			return Comparator.comparing(f -> Integer.parseInt(f.code.substring(f.code.indexOf('.') + 1)));
		}

		private static List<Entry> triples(List<String[]> splits) {
			return splits.stream().map(Entry::new).collect(toList());
		}

		private static Map<String, String> asMap(List<Entry> entries) {
			Map<String, String> map = new HashMap<>(entries.size());
			for (Entry entry : entries) map.put(entry.key, entry.value);
			return map;
		}

		private static Map<String, List<String[]>> triples(File file) {
			try {
				return Files.readAllLines(file.toPath()).stream()
						.filter(s -> !s.isEmpty())
						.map(s -> s.split("\t"))
						.collect(groupingBy(s -> s[0]));
			} catch (IOException ignored) {
				return Collections.emptyMap();
			}
		}

		private void setMarkers() {
			Field marker = null;
			for (Field field : fields) {
				if (field.type == Type.Marker || field.type == Type.Section) marker = field;
				else if (marker != null) field.marker(marker);
			}
		}

		public Field field(String key) {
			return fields.stream().filter(f -> f.code.equalsIgnoreCase(key) || f.name.equalsIgnoreCase(key)).findFirst().orElse(null);
		}

		public List<Field> fields() {
			return fields;
		}

		private void add(String code, List<Entry> entries) {
			add(new Field(code, asMap(entries)));
		}

		private void add(Field field) {
			fields.add(field);
		}

		@Override
		public Iterator<Field> iterator() {
			return fields.iterator();
		}

		public enum Type {
			String, Number, Date, Option, MultiOption, Image, Entity, Package, Note, Marker, Section, Validation, Signature
		}

		public static class Field {
			public final String name;
			public final Type type;
			private final String code;
			private final boolean optional;
			private final String conditional;
			private final List<String> filter;
			private final Map<String, String> entries;
			private Field marker;

			public Field(String code, Map<String, String> entries) {
				this.code = code;
				this.name = entries.get("name");
				this.type = Type.valueOf(entries.getOrDefault("type", "String"));
				this.entries = entries;
				this.optional = Boolean.parseBoolean(entries.getOrDefault("optional", "false"));
				this.filter = toList(entries.getOrDefault("filter", ""));
				this.conditional = entries.get("conditional");
			}

			public Map<String, String> entries() {
				return Collections.unmodifiableMap(entries);
			}

			public Annex annex() {
				int annexStart = name.indexOf('+');
				return annexStart < 0 ? null : getAnnex();
			}

			private Annex getAnnex() {
				return hasAnnexInstance() ? new AnnexInstance(annexName(), this) : new AnnexShared(annexName());
			}

			private boolean hasAnnexInstance() {
				return entries.getOrDefault("annex.type", Annex.Type.shared.name()).equals(Annex.Type.instance.name());
			}

			private String annexName() {
				int annexStart = name.indexOf('+');
				return annexStart < 0 ? null : name.substring(annexStart + 1);
			}

			public String title(String language) {
				return get("title.", language);
			}

			public String description(String language) {
				return get("description.", language);
			}

			public List<String> values(String language) {
				return Stream.of(get("values.", language).split(";")).map(String::trim).collect(Collectors.toList());
			}

			public Double valueMin() {
				return entries.containsKey("value-min") ? Double.parseDouble(entries.get("value-min")) : null;
			}

			public Double valueMax() {
				return entries.containsKey("value-max") ? Double.parseDouble(entries.get("value-max")) : null;
			}

			public Double valueDefault() {
				return entries.containsKey("value-default") ? Double.parseDouble(entries.get("value-default")) : 0;
			}

			public String unit() {
				return entries.get("unit");
			}

			public boolean isOptional() {
				return optional;
			}

			public boolean isConditional() {
				return conditional != null;
			}

			public String conditional() {
				return conditional;
			}

			public List<String> filter() {
				return filter;
			}

			public Field marker() {
				return marker;
			}

			public String get(String attribute, String language) {
				return entries.getOrDefault(attribute + language, "");
			}

			@Override
			public String toString() {
				return "Field{" + "type=" + type + ", name='" + name + '\'' + '}';
			}

			private void marker(Field marker) {
				this.marker = marker;
			}

			@Override
			public boolean equals(Object o) {
				if (this == o) return true;
				if (o == null || getClass() != o.getClass()) return false;
				Field field = (Field) o;
				return Objects.equals(name, field.name) && type == field.type;
			}

			@Override
			public int hashCode() {
				return Objects.hash(name, type);
			}

			static List<String> toList(String value) {
				if (value == null) return Collections.emptyList();
				return Stream.of(value.split(","))
						.map(String::trim)
						.filter(s -> !s.isEmpty())
						.collect(Collectors.toList());
			}
		}

		public static class Entry {
			public final String key;
			public final String value;

			public Entry(String[] split) {
				this(split[1], split[2]);
			}

			public Entry(String key, String value) {
				this.key = key;
				this.value = value;
			}
		}

		public static abstract class Annex {

			public final String name;

			Annex(String name) {
				this.name = name;
			}

			public Type type() {
				return (this instanceof AnnexInstance) ? Type.instance : Type.shared;
			}

			public enum Type {
				shared, instance
			}
		}

		public static class AnnexShared extends Annex {

			private AnnexShared(String name) {
				super(name);
			}
		}

		public static class AnnexInstance extends Annex {

			private static final String FIELD_ATTRIB_MARK = "$";

			public final String instanceName;
			private final Map<String, String> attributes;

			private AnnexInstance(String name, Checklist.Field check) {
				super(name);
				this.instanceName = name + "$" + check.code;
				this.attributes = check.entries().entrySet().stream()
						.collect(Collectors.toMap(
								e -> e.getKey().startsWith("annex.") ? e.getKey().replace("annex.", "") : FIELD_ATTRIB_MARK + e.getKey(),
								Map.Entry::getValue
						));
				this.attributes.put(FIELD_ATTRIB_MARK + "code", check.code);
				this.attributes.put(FIELD_ATTRIB_MARK + "name", check.name.replace("+" + name, ""));
				this.attributes.put(FIELD_ATTRIB_MARK + "index", check.code.substring(check.code.indexOf('.') + 1));
			}

			public Map<String, String> attributes(String language) {
				return attributes.entrySet().stream().filter(e -> matchesLanguageOrIsGlobal(e, language)).collect(Collectors.toMap(
						e -> e.getKey().replace("." + language, ""),
						Map.Entry::getValue
				));
			}

			private boolean matchesLanguageOrIsGlobal(Map.Entry<String, String> entry, String language) {
				int langStart = entry.getKey().lastIndexOf('.');
				if(langStart < 0) return true; // No language -> Global
				String attribLang = entry.getKey().substring(langStart + 1);
				if(Locale.forLanguageTag(attribLang) == null) return true; // No language -> Global
				return language.equals(attribLang);
			}
		}
	}
}
