package io.intino.monet.box.orders;

import io.intino.alexandria.logger.Logger;
import io.intino.monet.engine.Order;

import java.io.Closeable;
import java.io.File;
import java.sql.*;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import static java.lang.String.format;

@SuppressWarnings("SynchronizeOnNonFinalField")
public class OrdersDatabase implements Closeable {

	public static final String INSERT = "INSERT INTO orders (id, code, input, properties, createDate, scheduleDate, dueDate) Values ('%s','%s','%s','%s',%d,%d,%d);";
	public static final String UPDATE = "UPDATE orders SET input='%s', properties='%s', scheduleDate=%d, dueDate=%d WHERE id='%s';";
	public static final String DELETE = "DELETE FROM orders WHERE id='%s';";

	final File file;
	private Connection connection;
	private Statement modifyStatement;
	private PreparedStatement queryStatement;
	private Statement queryStatementForCount;
	private PreparedStatement requestQueryStatement;
	private int pendingTransactions = 0;
	private long lastActivity = Instant.now().toEpochMilli();

	private static final int timeThreshold = 5 * 1000;
	private static final int amountThreshold = 5000;

	public OrdersDatabase(File file) {
		this.file = file;
		try {
			this.connection = DriverManager.getConnection("jdbc:sqlite:" + file);
			this.connection.createStatement().execute("CREATE TABLE IF NOT EXISTS orders (id text NOT NULL PRIMARY KEY, code text, input text, properties text, createDate bigint, scheduleDate bigint, dueDate bigint);");
			this.connection.createStatement().execute("CREATE INDEX IF NOT EXISTS idx_orders ON orders (id);");
			this.modifyStatement = connection.createStatement();
			this.requestQueryStatement = connection.prepareStatement("SELECT * FROM orders WHERE id=?");
			this.queryStatement = connection.prepareStatement("SELECT * FROM orders");
			this.queryStatementForCount = connection.createStatement();
			this.connection.setAutoCommit(false);
		} catch (SQLException e) {
			Logger.error(e);
		}
	}

	public synchronized void commit() {
		if (pendingTransactions == 0 || (Instant.now().toEpochMilli() - lastActivity) < timeThreshold) return;
		doCommit();
	}

	public synchronized void add(Order order) {
		try {
			Long createDate = order.createDate() != null ? order.createDate().toEpochMilli() : null;
			Long scheduleDate = order.scheduleDate() != null ? order.scheduleDate().toEpochMilli() : null;
			Long dueDate = order.dueDate() != null ? order.dueDate().toEpochMilli() : null;
			String input = order.input() != null ? order.input() : "{}";
			String properties = order.properties() != null ? order.properties() : "{}";
			modifyStatement.execute(format(INSERT, order.id(), order.code(), input, properties, createDate, scheduleDate, dueDate));
			updatePending();
		} catch (SQLException e) {
			Logger.error(e);
		}
	}

	public synchronized void update(Order order) {
		try {
			Long scheduleDate = order.scheduleDate() != null ? order.scheduleDate().toEpochMilli() : null;
			Long dueDate = order.dueDate() != null ? order.dueDate().toEpochMilli() : null;
			String input = order.input() != null ? order.input() : "{}";
			String properties = order.properties() != null ? order.properties() : "{}";
			modifyStatement.execute(format(UPDATE, input, properties, scheduleDate, dueDate, order.id()));
			updatePending();
		} catch (SQLException e) {
			Logger.error(e);
		}
	}

	public synchronized void delete(String id) {
		try {
			modifyStatement.execute(format(DELETE, id));
			updatePending();
		} catch (SQLException e) {
			Logger.error(e);
		}
	}

	public synchronized List<Order> getAll() {
		try (ResultSet rs = queryStatement.executeQuery()) {
			return ordersOf(rs);
		} catch (SQLException e) {
			Logger.error(e);
			return Collections.emptyList();
		}
	}

	public synchronized Order get(String id) {
		try {
			requestQueryStatement.setString(1, id);
			requestQueryStatement.execute();
			List<Order> orderList = ordersOf(requestQueryStatement.getResultSet());
			return orderList.isEmpty() ? null : orderList.get(0);
		}
		catch(SQLException e) {
			Logger.error(e);
			return null;
		}
	}

	public synchronized int count() {
		String query = "SELECT count(*) AS total FROM orders";
		try (ResultSet resultSet = queryStatementForCount.executeQuery(query)) {
			return resultSet.getInt("total");
		} catch (SQLException e) {
			Logger.error("Query: " + query, e);
			return 0;
		}
	}

	private List<Order> ordersOf(ResultSet rs) throws SQLException {
		List<Order> result = new ArrayList<>();
		while (rs.next()) result.add(orderOf(rs));
		return result;
	}

	private Order orderOf(ResultSet rs) throws SQLException {
		Order result = new Order(rs.getString(1), rs.getString(2));
		result.input(rs.getString(3));
		result.properties(rs.getString(4));
		result.createDate(Instant.ofEpochMilli(rs.getLong(5)));
		result.scheduleDate(rs.getLong(6) != 0 ? Instant.ofEpochMilli(rs.getLong(6)) : null);
		result.dueDate(rs.getLong(7) != 0 ? Instant.ofEpochMilli(rs.getLong(7)) : null);
		return result;
	}

	public void close() {
		try {
			if (connection.isClosed()) return;
			doCommit();
			modifyStatement.close();
			requestQueryStatement.close();
			queryStatement.close();
			queryStatementForCount.close();
			connection.close();
		} catch (SQLException e) {
			Logger.error(e);
		}
	}

	private void updatePending() {
		pendingTransactions++;
		lastActivity = Instant.now().toEpochMilli();
		if (pendingTransactions >= amountThreshold) doCommit();
	}

	private synchronized void doCommit() {
		try {
			connection.commit();
			pendingTransactions = 0;
		} catch (SQLException e) {
			Logger.error(e);
		}
	}

	private String clean(String value) {
		return value != null ? value.replace("'", "''") : "";
	}

}
