package io.intino.monet.box.util;

import io.intino.alexandria.Scale;
import io.intino.alexandria.Timetag;
import io.intino.monet.box.MonetBox;
import io.intino.monet.box.ui.displays.DisplayHelper;
import io.intino.monet.engine.Order;
import io.intino.monet.engine.OrderTypes;
import io.intino.monet.engine.edition.*;
import io.intino.monet.engine.edition.editors.*;

import java.io.File;
import java.io.IOException;
import java.time.LocalDate;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import static java.util.Collections.emptyList;
import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toMap;

public class FormHelper {

    public static Form create(MonetBox box, Order order) {
        return new Form(formDefinition(box, order), store(box, order));
    }

    public static FormDefinition formDefinition(MonetBox box, Order order) {
        return new FormDefinition(fields(box, order).toArray(new FieldDefinition[0]));
    }

    public static Map<String, List<String>> values(Form form) {
        return form.editions().stream().filter(e -> !e.isHidden()).collect(toMap(Edition::name, FormHelper::valuesOf));
    }

    public static String valueOf(Edition edition) {
        return valueOf(valuesOf(edition));
    }

    public static List<String> valuesOf(Edition edition) {
        if (edition instanceof SectionEdition)
            return singletonList(String.valueOf(!edition.as(SectionEdition.class).isEmpty() || !edition.isOptional()));
        if (edition.isEmpty() || edition.isNull()) return emptyList();
        if (edition instanceof DateEdition) {
            LocalDate localDate = edition.as(DateEdition.class).get();
            return localDate != null ? singletonList(new Timetag(localDate, Scale.Day).value()) : emptyList();
        }
        if (edition instanceof ImageEdition) {
            ImageEdition.Image image = edition.as(ImageEdition.class).get();
            return image != null ? singletonList(image.label) : emptyList();
        }
        if (edition instanceof NoteEdition) {
            NoteEdition.Note note = edition.as(NoteEdition.class).get();
            return note != null ? singletonList(note.label) : emptyList();
        }
        if (edition instanceof NumberEdition)
            return singletonList(String.valueOf(edition.as(NumberEdition.class).get()));
        if (edition instanceof OptionEdition) return singletonList(edition.as(OptionEdition.class).get());
        if (edition instanceof OptionMultipleEdition)
            return Arrays.asList(edition.as(OptionMultipleEdition.class).get());
        if (edition instanceof StringEdition) return singletonList(edition.as(StringEdition.class).get());
        return emptyList();
    }

    public static String valueOf(List<String> values) {
        return String.join(";", values);
    }

    private static List<FieldDefinition> fields(MonetBox box, Order order) {
        return checkListTriples(order).entrySet().stream().map(e -> new FieldDefinition(e.getKey().name, triplesOf(box, e))).collect(Collectors.toList());
    }

    private static Map<String, String> triplesOf(MonetBox box, Map.Entry<OrderTypes.Checklist.Field, Map<String, String>> entry) {
        Map<String, String> result = new HashMap<>(entry.getValue());
        result.put("attachment-repository", box.archetype().definitions().orderTypes().attachmentsDirectory().getAbsolutePath());
        result.put("thesaurus-repository", box.archetype().definitions().orderTypes().thesaurusDirectory().getAbsolutePath());
        return result;
    }

    private static Map<OrderTypes.Checklist.Field, Map<String, String>> checkListTriples(Order order) {
        return DisplayHelper.checkListTriples(order);
    }

    private static FormStore store(MonetBox box, Order order) {
        File directory = box.archetype().repository().workorders().getStoreDirectory(order.store());
        LocalFormStore result = new LocalFormStore(directory);
        try {
            result.load();
        } catch (IOException ignored) {
        }
        return result;
    }

}
