package io.intino.monet.box.workreports;

import io.intino.alexandria.office.DocxBuilder;
import io.intino.alexandria.office.components.ImageView;
import io.intino.alexandria.office.components.Paragraph;

import java.io.File;
import java.io.IOException;
import java.util.*;

public class WorkReportTemplate {

    private final String name;
    private final File file;
    private final DocxBuilder docxBuilder;
    private boolean empty = true;
    private final Map<String, Paragraph> fields = new LinkedHashMap<>();
    private final Set<String> images = new LinkedHashSet<>();
    private boolean isMain = false;
    private final Map<String, String> extraInfo = new HashMap<>(1);

    public WorkReportTemplate(String name, File templateFile) {
        this.name = name;
        if(!templateFile.exists()) throw new IllegalArgumentException("Template file " + templateFile + " does not exist");
        this.file = templateFile;
        this.docxBuilder = DocxBuilder.create(templateFile);
    }

    public String name() {
        return name;
    }

    public File file() {
        return file;
    }

    public boolean contains(String field) {
        return fields.containsKey(field);
    }

    public boolean containsImage(String field) {
        return images.contains(field);
    }

    public WorkReportTemplate set(String field, String value) {
        if(value == null) return this;
        fields.put(field, new Paragraph().text(value));
        docxBuilder.replace(field, escape(value));
        empty &= value.isEmpty();
        return this;
    }

    public WorkReportTemplate set(String field, Paragraph value) {
        if(value == null) return this;
        fields.put(field, value);
        docxBuilder.replace(field, value);
        empty = false;
        return this;
    }

    public WorkReportTemplate set(String field, ImageView value) {
        if(value == null) return this;
        images.add(field);
        docxBuilder.replace(field, value);
        empty = false;
        return this;
    }

    public void save(File file) throws IOException {
        docxBuilder.save(file);
    }

    public boolean empty() {
        return empty;
    }

    public Map<String, String> extraInfo() {
        return extraInfo;
    }

    @Override
    public String toString() {
        return "WorkReportTemplate:" + file.toString();
    }

    public boolean isMainTemplate() {
        return isMain;
    }

    void setAsMainTemplate() {
        isMain = true;
    }

    private String escape(String value) {
        if (value == null || value.isEmpty()) return "";
        return "<![CDATA[" + value + "]]>";
    }

}
