package io.intino.monet.box.workreports;

import io.intino.alexandria.Json;
import io.intino.alexandria.logger.Logger;
import io.intino.alexandria.office.PdfBuilder;
import io.intino.alexandria.office.PdfDecorator;
import io.intino.alexandria.office.PdfMerger;
import io.intino.alexandria.office.PdfMetadata;
import io.intino.monet.box.WorkReportGenerator;
import io.intino.monet.engine.WorkReport;
import org.apache.poi.openxml4j.util.ZipSecureFile;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.time.Instant;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import static java.nio.file.StandardCopyOption.REPLACE_EXISTING;

public class PdfWorkReportGenerator {

	private final WorkReportGenerator.Archetype archetype;
	private final File destination;
	private WorkReportBuilder builder;
	private WorkReportInfoProvider infoProvider;

	public PdfWorkReportGenerator(WorkReportGenerator.Archetype archetype, File destination) {
		this.archetype = archetype;
		this.destination = destination;
		this.builder = new WorkReportBuilder(archetype);
		this.infoProvider = new WorkReportInfoProvider();
	}

	public PdfWorkReportGenerator builder(WorkReportBuilder builder) {
		this.builder = builder;
		return this;
	}

	public PdfWorkReportGenerator infoProvider(WorkReportInfoProvider provider) {
		this.infoProvider = provider;
		return this;
	}

	public void generate(WorkReport workReport) {
		try {
			Map<String, WorkReportTemplate> templates = setWorkReportInformation(workReport);
			List<File> pdfDocuments = generateTempPdfDocuments(templates);
			mergeAndGenerateTheFinalWorkReportDocument(pdfDocuments);
		} catch (Throwable e) {
			throw new RuntimeException(e);
		}
	}

	private void mergeAndGenerateTheFinalWorkReportDocument(List<File> pdfDocuments) throws IOException {
		try {
			if(pdfDocuments.size() == 1)
				moveToDestination(pdfDocuments.get(0));
			else
				mergeIntoTheFinalDocument(pdfDocuments);

			setMetadata();
		} finally {
			pdfDocuments.forEach(File::delete);
		}
	}

	private void setMetadata() {
		try {
			WorkReport workReport = builder.workReport();

			PdfMetadata metadata = new PdfMetadata(destination);

			metadata.setCreator("monet");
			metadata.setProducer("monet");
			metadata.setAuthor(workReport.order().property("author")); // TODO: check

			metadata.setCustomValue("input", workReport.order().input());
			metadata.setCustomValue("checklist", Json.toString(workReport.attributes()));
			// TODO: more info?

			new PdfDecorator(destination).setMetadata(metadata).save(destination);

		} catch (Exception e) {
			Logger.debug("Could not add metadata to pdf: " + e.getClass().getSimpleName() + " => " + e.getMessage());
		}
	}

	private void mergeIntoTheFinalDocument(List<File> pdfDocuments) throws IOException {
		PdfMerger merger = new PdfMerger();
		pdfDocuments.forEach(merger::append);
		merger.merge(destination);
	}

	private void moveToDestination(File file) throws IOException {
		Files.move(file.toPath(), destination.toPath(), REPLACE_EXISTING);
	}

	private List<File> generateTempPdfDocuments(Map<String, WorkReportTemplate> templates) {
		return templates.values().stream().map(this::generateDocumentFor).filter(Objects::nonNull).collect(Collectors.toList());
	}

	private File generateDocumentFor(WorkReportTemplate template) {
		// Preventing Zip bomb exception, since the docx files comes from a secure source
		ZipSecureFile.setMinInflateRatio(0);

		File tmpFile = new File(archetype.tmp(), currentInstant() + ".docx");
		File pdfFile = new File(archetype.tmp(), currentInstant() + ".pdf");
		if(template.empty()) return null;

		saveTemplateToPdf(template, tmpFile, pdfFile);

		return pdfFile;
	}

	private static void saveTemplateToPdf(WorkReportTemplate template, File tmpFile, File pdfFile) {
		try {
			template.save(tmpFile);
			PdfBuilder.create(tmpFile).save(pdfFile);
		} catch (Exception e) {
			throw new RuntimeException("Error while generating pdf of " + template.file() + ": " + e.getMessage(), e);
		} finally {
			tmpFile.delete();
		}
	}

	private String currentInstant() {
		return Instant.now().toString().replace(":", "");
	}

	private Map<String, WorkReportTemplate> setWorkReportInformation(WorkReport workReport) throws WorkReportBuilder.WorkReportException {
		infoProvider.workReport(workReport);
		return builder.workReport(workReport).infoProvider(infoProvider).build();
	}
}
