package io.intino.monet.box.commands.order;

import io.intino.alexandria.Resource;
import io.intino.alexandria.logger.Logger;
import io.intino.alexandria.zip.Zip;
import io.intino.monet.box.MonetBox;
import io.intino.monet.box.WorkReportGenerator;
import io.intino.monet.box.WorkReportGenerator.Archetype.OrderTypes;
import io.intino.monet.box.commands.Command;
import io.intino.monet.box.util.OrderHelper;
import io.intino.monet.box.util.WorkReportHelper;
import io.intino.monet.box.workreports.WorkReportInfoProvider;
import io.intino.monet.engine.Order;
import io.intino.monet.engine.WorkReport;
import io.intino.monet.engine.edition.Edition;
import io.intino.monet.engine.edition.Field;
import io.intino.monet.engine.edition.FieldType;
import io.intino.monet.engine.edition.Form;
import io.intino.monet.engine.edition.editors.ImageEdition;
import io.intino.monet.engine.edition.editors.NoteEdition;
import io.intino.monet.engine.edition.editors.PackageEdition;
import io.intino.monet.engine.edition.editors.SignatureEdition;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.StandardOpenOption;
import java.time.Instant;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import static java.util.stream.Collectors.toList;

public class ResolveOrderCommand extends Command<File> {
	public Order order;
	public boolean testEnvironment;
	public String language;
	public Form form;
	public Instant stampDate;

	public ResolveOrderCommand(MonetBox box) {
		super(box);
	}

	public static boolean canExecute(MonetBox box, Order order, Form form, String author) {
		WorkReportGenerator generator = box.workReportGenerator();
		return generator.canGenerate(workReport(order, attributes(box, order, form), author), form);
	}

	public File executePreview() {
		return generate(workReport(order, attributes(), images(), language, author()));
	}

	@Override
	public File execute() {
		WorkReport report = workReport(order, attributes(), images(), language, author());
		File result = generate(report);
		savePackageFiles(result, report);
		box.orderApi().orderFinished(order, report, result);
		box.orderApi().delete(order);
		return result;
	}

	private File generate(WorkReport report) {
		WorkReportGenerator generator = box.workReportGenerator();
		if (generator == null) return null;
		return generator.generate(report, form, testEnvironment);
	}

	private Map<String, String> attributes() {
		return attributes(box, order, form);
	}

	private static Map<String, String> attributes(MonetBox box, Order order, Form form) {
		Map<String, String> result = OrderHelper.valuesOf(order, form);
		return WorkReportHelper.addCalculationsTo(result, order, OrderTypes.wrap(box.archetype().definitions().orderTypes()));
	}

	private void savePackageFiles(File result, WorkReport report) {
		try {
			if (result == null) return;
			String uploadFileId = order.inputMap().getOrDefault("upload", null);
			if (uploadFileId == null) return;
			File uploadZipFile = box.archetype().repository().uploads().getUploadZipFile(uploadFileId);
			List<File> files = packageFiles();
			Zip zip = new Zip(uploadZipFile);
			zip.write(packageFilename(order, report), new FileInputStream(result));
			files.forEach(f -> {
				try {
					zip.write(f.getName(), Files.newInputStream(f.toPath()), StandardOpenOption.CREATE);
				} catch (IOException e) {
					Logger.error(e);
				}
			});
		} catch (IOException e) {
			Logger.error(e);
		}
	}

	private String packageFilename(Order order, WorkReport report) {
		io.intino.monet.engine.OrderTypes.Record record = io.intino.monet.engine.OrderTypes.of(order.code());
		if (record == null) return String.format(DefaultFilename, order.code());
		String pattern = record.reportFilename();
		String filename = pattern != null ? OrderHelper.replaceVariables(pattern, attributesWithBasicInfo(record, report)) : defaultFilename(record);
		return filename.replace(":", "") + ".pdf";
	}

	private static final String DefaultFilename = "%s-work-report";
	private String defaultFilename(io.intino.monet.engine.OrderTypes.Record record) {
		String label = record.label(language);
		return label != null ? label : String.format(DefaultFilename, order.code());
	}

	private Map<String, String> attributesWithBasicInfo(io.intino.monet.engine.OrderTypes.Record record, WorkReport report) {
		Map<String, String> attributes = attributes();
		attributes.put("label", record.label(language));
		attributes.putAll(box.workReportInfoProvider().workReport(report).info());
		return attributes;
	}

	private List<Resource> images() {
		List<Resource> result = form.editions(FieldType.Image).stream().filter(this::isVisible).map(this::resourceOf).collect(toList());
		result.addAll(form.editions(FieldType.Note).stream().filter(this::isVisible).map(this::resourceOf).collect(toList()));
		result.addAll(form.editions(FieldType.Signature).stream().filter(this::isVisible).map(this::resourceOf).collect(toList()));
		return result;
	}

	private boolean isVisible(Edition edition) {
		return !isInvisible(edition);
	}

	private static boolean isInvisible(Edition edition) {
		return edition.isHidden() || edition.isSkipped() || edition.isDisabled();
	}

	private Resource resourceOf(Edition edition) {
		if (edition instanceof ImageEdition) return resourceOf(edition.as(ImageEdition.class));
		else if (edition instanceof NoteEdition) return resourceOf(edition.as(NoteEdition.class));
		else if (edition instanceof SignatureEdition) return resourceOf(edition.as(SignatureEdition.class));
		return new Resource(edition.name(), emptyImageData());
	}

	private Resource resourceOf(ImageEdition edition) {
		ImageEdition.Image image = edition.get();
		if (image != null && image.file != null && image.file.exists()) return OrderHelper.resourceOf(image.file);
		return new Resource(edition.name(), emptyImageData());
	}

	private Resource resourceOf(NoteEdition edition) {
		NoteEdition.Note note = edition.get();
		if (note != null && note.file != null && note.file.exists()) return OrderHelper.resourceOf(note.file);
		return new Resource(edition.name(), emptyImageData());
	}

	private Resource resourceOf(SignatureEdition edition) {
		SignatureEdition.Signature signature = edition.get();
		if (signature != null && signature.file != null && signature.file.exists())
			return OrderHelper.resourceOf(signature.file);
		return new Resource(edition.name(), emptyImageData());
	}

	private byte[] emptyImageData() {
		return new byte[0];
	}

	private List<File> packageFiles() {
		return form.editions(FieldType.Package).stream().filter(f -> !f.isHidden()).map(e -> ((PackageEdition) e).get().files()).flatMap(Collection::stream).collect(toList());
	}

	private static WorkReport workReport(Order order, Map<String, String> attributes, String author) {
		return workReport(order, attributes, Collections.emptyList(), "en", author);
	}

	private static WorkReport workReport(Order order, Map<String, String> attributes, List<Resource> images, String language, String author) {
		WorkReport result = new WorkReport();
		result.order(order);
		result.author(author);
		result.attributes(attributes);
		result.images(images);
		result.language(language);
		result.finished(Instant.now());
		return result;
	}

}