package io.intino.monet.box.workreports;

import io.intino.alexandria.Resource;
import io.intino.alexandria.logger.Logger;
import io.intino.alexandria.office.Image;
import io.intino.alexandria.office.ImageView;
import io.intino.monet.box.ImageViewFactory;
import io.intino.monet.box.WorkReportGenerator;
import io.intino.monet.box.util.Formatters;
import io.intino.monet.box.util.WorkReportHelper;
import io.intino.monet.engine.CheckListProvider;
import io.intino.monet.engine.Order;
import io.intino.monet.engine.OrderTypes;
import io.intino.monet.engine.OrderTypes.Checklist;
import io.intino.monet.engine.WorkReport;

import java.io.File;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

public class WorkReportBuilder {

    private static final String MAIN_TEMPLATE_NAME = "main";

    private final WorkReportGenerator.Archetype archetype;
    private WorkReport workReport;

    private Checklist checklist;
    private final Map<String, WorkReportTemplate> templates = new LinkedHashMap<>();

    private ImageViewFactory imageViewFactory = new ImageViewFactory.Default();

    public WorkReportBuilder(WorkReportGenerator.Archetype archetype) {
        this.archetype = archetype;
    }

    public Map<String, WorkReportTemplate> build() throws WorkReportException {
        validate();
        templates.clear();
        buildDocuments();
        return templates;
    }

    private void buildDocuments() throws WorkReportException {
        createMainTemplate();
        createSecondaryTemplates();
        fillWorkReportTemplates();
    }

    private void fillWorkReportTemplates() {
        setInputs(mainTemplate());
        setBasicInfo(mainTemplate());
        setChecksAndCalculations();
        setImages();
    }

    protected void setChecksAndCalculations() {

        Map<String, String> workReportChecks = withCalculations();

        for(Map.Entry<String, String> checkEntry : workReportChecks.entrySet()) {

            String name = checkEntry.getKey();

            Checklist.Field check = getCheck(name);

            WorkReportTemplate template = check != null ? templateOf(check) : templateOf(name);

            if(template == null) {
                Logger.warn("Template of " + name + " not found...");
                return;
            }

            if (!template.contains(checkName(name))) template.set(checkName(name), checkEntry.getValue());
        }

        setDefaultValuesForNotProcessedChecks(workReportChecks);
    }

    private void setDefaultValuesForNotProcessedChecks(Map<String, String> workReportChecks) {
        Set<String> notProcessedChecks = checklist.fields().stream().map(f -> f.name).collect(Collectors.toSet());
        notProcessedChecks.removeAll(workReportChecks.keySet());

        WorkReportTemplate template = mainTemplate();
        for(String check : notProcessedChecks) {
            template.set(check, "");
            // TODO: handle images
        }
    }

    private Checklist.Field getCheck(String name) {
        return checklist.fields().stream().filter(f -> f.name.equals(name)).findFirst().orElse(null);
    }

    protected void setBasicInfo(WorkReportTemplate mainTemplate) {
        mainTemplate.set("date", Formatters.textDate(workReport.finished(), workReport.language()));
        mainTemplate.set("comments", workReport.comments() == null ? "" : workReport.comments());
    }

    private String valueOrEmpty(String value) {
        return value != null ? value : "";
    }

    protected void setInputs(WorkReportTemplate mainTemplate) {
        Order order = workReport.order();
        if (order.inputMap() != null) order.inputMap().forEach(mainTemplate::set);
    }

    private Map<String, String> withCalculations() {
        return WorkReportHelper.addCalculationsTo(workReport.attributes(), workReport.order(), archetype.orderTypes());
    }

    protected void setImages() {
        for (Resource image : workReport.images()) setImage(image);
    }

    protected void setImage(Resource image) {
        try {
            ImageView imageView = imageViewFactory.create(new Image(image.bytes()));
            WorkReportTemplate template = templateOf(getCheck(image.name()));
            if(template == null) {
                Logger.warn("Template of " + image.name() + " not found...");
                return;
            }
            template.set(checkName(image.name()), imageView);
        } catch (Throwable e) {
            Logger.error("Failed to create image " + image.name() + ": " + e.getMessage(), e);
        }
    }

    private void createMainTemplate() {
        WorkReportTemplate template = new WorkReportTemplate(WorkReportHelper.template(
                archetype.orderTypes(),
                workReport.area(), workReport.order(),
                workReport.language()));

        templates.put(MAIN_TEMPLATE_NAME, template);
    }

    private void createSecondaryTemplates() throws WorkReportException {
        Order order = workReport.order();
        OrderTypes.Record orderType = OrderTypes.of(order.code());
        if(orderType == null) throw new WorkReportException("OrderType not found for order " + order.code());
        for(String annex : orderType.annexes()) {
            String[] entry = annex.split(":");
            if(entry.length != 2) continue; // Annex is malformed, it should be {id}:{file}.
            templates.put(entry[0].trim(), templateOfFilename(entry[1].trim()));
        }
    }

    private WorkReportTemplate templateOf(Checklist.Field check) {
        return templateOf(check.name);
    }

    private WorkReportTemplate templateOf(String name) {
        if(!name.contains("+")) return mainTemplate();
        return templates.get(name.substring(name.indexOf('+') + 1));
    }

    private WorkReportTemplate templateOfFilename(String filename) {
        return new WorkReportTemplate(
                new File(mainTemplate().file().getAbsolutePath().replace(".docx", "$" + filename + ".docx")));
    }

    private WorkReportTemplate mainTemplate() {
        return templates.get(MAIN_TEMPLATE_NAME);
    }

    private String checkName(String name) {
        return !name.contains("+") ? name : name.substring(0, name.indexOf('+'));
    }

    protected void validate() throws WorkReportException {
        if(workReport == null) throw new WorkReportException("WorkReport is null");
        if(workReport.order() == null) throw new WorkReportException("WorkReport orderId is null");
        if(workReport.language() == null) throw new WorkReportException("Language is not specified");
        if((checklist = new CheckListProvider().findCheckList(workReport.order().code())) == null) throw new WorkReportException("No checklist found for order " + workReport.order().code());
    }

    @SuppressWarnings("unchecked")
    public <T extends WorkReport> T workReport() {
        return (T) workReport;
    }

    public WorkReportBuilder workReport(WorkReport workReport) {
        this.workReport = workReport;
        return this;
    }

    public WorkReportBuilder imageViewFactory(ImageViewFactory imageViewFactory) {
        this.imageViewFactory = imageViewFactory == null ? new ImageViewFactory.Default() : imageViewFactory;
        return this;
    }

    public static class WorkReportException extends Exception {
        public WorkReportException(String message) {
            super(message);
        }
    }

}
