package io.intino.monet.box.ui.displays.templates;

import io.intino.alexandria.Json;
import io.intino.alexandria.ui.displays.events.AddItemEvent;
import io.intino.alexandria.ui.displays.events.SelectionEvent;
import io.intino.monet.box.MonetBox;
import io.intino.monet.box.ui.datasources.OrderTypesDatasource;
import io.intino.monet.box.ui.displays.items.OrderTypeItem;
import io.intino.monet.engine.Order;
import io.intino.monet.engine.OrderTypes;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import java.util.stream.Collectors;

public class OrderTypesTemplate extends AbstractOrderTypesTemplate<MonetBox> {
    private OrderTypes.Record selected;
    private static final Map<String, String> inputMap = new HashMap<>();

    public OrderTypesTemplate(MonetBox box) {
        super(box);
    }

    @Override
    public void init() {
        super.init();
        orderTypeItems.onSelect(this::open);
        orderTypeItems.onAddItem(this::refresh);
        orderTypeItems.source(new OrderTypesDatasource(box(), session()));
        orderTypeItems.reload();
        openWizard.onExecute(e -> {
            inputDialogBox.close();
            openOrderTypeWizard();
        });
        refresh.onExecute(e -> orderTypeItems.reload());
        inputDialogBox.onOpen(e -> refreshInputDialog());
        orderTypeDialogBox.onOpen(e -> refreshOrderTypeDialog());
    }

    private void refresh(AddItemEvent event) {
        OrderTypes.Record orderType = event.item();
        OrderTypeItem display = event.component();
        display.code.value(orderType.code() + ": " + translate(orderType.target().name()));
        display.label.value(orderType.label(language()));
        display.properties.value(properties(orderType).entrySet().stream().map(e -> "<b>" + e.getKey() + "</b>&nbsp;" + e.getValue()).collect(Collectors.joining("&nbsp;&nbsp;")));
    }

    private Map<String, String> properties(OrderTypes.Record orderType) {
        Map<String, String> properties = new HashMap<>();
        properties.put("manual", translate(orderType.isManual() ? "Yes" : "No"));
        properties.put("effort", String.valueOf(orderType.effort()));
        if (orderType.channel() != null) properties.put("channel", orderType.channel().name());
        if (orderType.category() != null) properties.put("category", orderType.category());
        if (!orderType.annexes().isEmpty()) properties.put("annexes", String.join(", ", orderType.annexes()));
        return properties;
    }

    private void refreshInputDialog() {
        inputs.clear();
        selected.input().forEach(i -> fill(i, inputs.add()));
    }

    private void refreshOrderTypeDialog() {
        if (selected == null) return;
        orderTypeDialogBox.title(OrderTypes.of(selected.code()).label(language()));
        Order order = new Order("TEST-" + selected.code(), selected.code()).input(Json.toString(orderInputs()));
        orderStamp.order(order);
        orderStamp.onComplete((file, form) -> {
            if (file.exists()) file.delete();
            orderTypeDialogBox.close();
        });
        orderStamp.testEnvironment(true);
        orderStamp.refresh();
    }

    private void fill(String input, OrderInputEditor display) {
        display.input(input, inputMap.getOrDefault(input, null));
        display.onChange(value -> inputMap.put(input, value));
        display.refresh();
    }

    private void open(SelectionEvent event) {
        if (event.selection().size() <= 0) return;
        selected = (OrderTypes.Record) event.selection().get(0);
        orderTypeItems.selection(new ArrayList<>());
        if (selected == null) return;
        if (selected.isManual()) openOrderTypeWizard();
        else openInputDialog();
    }

    private void openInputDialog() {
        inputDialogBox.open();
    }

    private void openOrderTypeWizard() {
        orderTypeDialogBox.open();
    }

    private Map<String, String> orderInputs() {
        Map<String, String> result = new HashMap<>();
        result.put("upload", UUID.randomUUID().toString());
        if (selected.isManual()) return result;
        selected.input().forEach(i -> result.put(i, inputMap.getOrDefault(i, "")));
        return result;
    }

}