package io.intino.monet.box;

import io.intino.alexandria.logger.Logger;
import io.intino.monet.archetype.Archetype;
import io.intino.monet.box.orders.OrdersDatabase;
import io.intino.monet.engine.Order;
import io.intino.monet.engine.WorkReport;
import org.apache.commons.io.FileUtils;

import java.io.File;
import java.io.IOException;
import java.util.List;

public class OrderApi {
	private final Archetype archetype;
	private final OrdersDatabase ordersDatabase;
	private OrderFinishedListener orderFinishedListener;

	public OrderApi(Archetype archetype) {
		this.archetype = archetype;
		this.ordersDatabase = new OrdersDatabase(archetype.datamarts().monet().getOrders());
	}

	public void onOrderFinished(OrderFinishedListener listener) {
		this.orderFinishedListener = listener;
	}

	public List<Order> orders() {
		return ordersDatabase.getAll();
	}

	public int count() {
		return ordersDatabase.count();
	}

	public Order order(String id) {
		return ordersDatabase.get(id);
	}

	public Order create(String id, String code, String input) {
		return create(new Order(id, code).input(input));
	}

	public Order create(Order order) {
		ordersDatabase.add(order);
		return order;
	}

	public Order update(Order order) {
		ordersDatabase.update(order);
		return order;
	}

	public void delete(String orderId) {
		delete(order(orderId));
	}

	public void delete(Order order) {
		if (order == null) return;
		ordersDatabase.delete(order.id());
		deleteStore(order);
	}

	public void orderFinished(Order order, WorkReport report, File reportFile) {
		if (orderFinishedListener == null) return;
		orderFinishedListener.accept(new OrderFinishedListener.OrderFinishedEvent().order(order).report(report).reportFile(reportFile));
	}

	public void stop() {
		ordersDatabase.close();
	}

	public void commit() {
		ordersDatabase.commit();
	}

	private void deleteStore(Order order) {
		try {
			if (order == null || order.isStoreDefined()) return;
			File storeDirectory = archetype.repository().workorders().getStoreDirectory(order.store());
			if (!storeDirectory.exists()) return;
			FileUtils.deleteDirectory(storeDirectory);
		} catch (IOException e) {
			Logger.error(e);
		}
	}

}
