package io.intino.consul.framework.utils;

import io.intino.alexandria.logger.Logger;
import io.intino.consul.framework.Activity;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.InetAddress;
import java.net.URL;
import java.net.UnknownHostException;
import java.util.PropertyResourceBundle;
import java.util.jar.Attributes;
import java.util.jar.JarEntry;
import java.util.jar.JarInputStream;
import java.util.jar.Manifest;

public class Utils {
	public static String calculateHostName() {
		try {
			return InetAddress.getLocalHost().getHostName().replace(".", "-");
		} catch (UnknownHostException e) {
			Logger.error(e.getMessage());
			return "consul-default-server";
		}
	}

	public static boolean isNumber(String a) {
		try {
			Double.parseDouble(a);
			return true;
		} catch (NumberFormatException e) {
			return false;
		}
	}

	public static long inMb(long value) {
		return value / (1024 * 1024);
	}

	private static double asPercent(double number) {
		return Math.floor(number * 100) / 100;
	}

	public static String currentVersion() {
		try {
			String file = Utils.class.getProtectionDomain().getCodeSource().getLocation().getFile();
			if (isWindows()) {
				file = file.replace("%20", " ");
				if (file.startsWith("/")) file = file.substring(1);
			}
			if (new File(file).isFile()) {
				URL url = new URL("jar:file:" + file + "!/META-INF/MANIFEST.MF");
				try (InputStream is = url.openStream()) {
					Manifest manifest = new Manifest(is);
					String version = manifest.getMainAttributes().getValue("Implementation-Version");
					if (version != null) return version.trim();
				}
			}
		} catch (IOException e) {
			Logger.error(e);
		}
		return "1.0.0";
	}

	public static Coors coors(Activity.System system, File mainArtifact) {
		if (!mainArtifact.exists()) return null;
		try (JarInputStream is = new JarInputStream(system.operatingSystem().fileSystem().openFile(mainArtifact.getPath()))) {
			Manifest manifest = is.getManifest();
			Attributes attributes = manifest.getMainAttributes();
			if (attributes.getValue("Package") != null && attributes.getValue("Implementation-Title") != null && attributes.getValue("Implementation-Version") != null)
				return new Coors(attributes.getValue("Package"), attributes.getValue("Implementation-Title"), attributes.getValue("Implementation-Version"));
			else {
				JarEntry jarEntry = findJarEntry(is);
				if (jarEntry != null) {
					PropertyResourceBundle bundle = new PropertyResourceBundle(is);
					return new Coors(bundle.getString("groupId"), bundle.getString("artifactId"), bundle.getString("version"));
				}
			}
			return new Coors("", attributes.getValue("Implementation-Title"), attributes.getValue("Implementation-Version"));
		} catch (Exception e) {
			Logger.error(e.getMessage());
		}
		return new Coors("", mainArtifact.getName().replace(".jar", ""), "1.0.0");
	}


	private static JarEntry findJarEntry(JarInputStream is) throws IOException {
		JarEntry entry = is.getNextJarEntry();
		while (entry != null && !entry.getName().endsWith("/pom.properties")) entry = is.getNextJarEntry();
		return entry;
	}

	public record Coors(String groupId, String artifactId, String version) {
		public String name() {
			return groupId + ":" + artifactId;
		}

		public String id() {
			return name() + ":" + version;
		}

	}


	public enum OS {osx, windows, linux}

	public static boolean isWindows() {
		return (System.getProperty("os.name").toLowerCase().contains("win"));
	}

	public static boolean isMac() {
		return (System.getProperty("os.name").toLowerCase().contains("mac"));
	}

	public static boolean isUnix() {
		return (System.getProperty("os.name").toLowerCase().contains("nix") || System.getProperty("os.name").toLowerCase().contains("nux") || System.getProperty("os.name").toLowerCase().indexOf("aix") > 0);
	}

	public static boolean isSolaris() {
		return (System.getProperty("os.name").toLowerCase().contains("sunos"));
	}

	public static OS operatingSystem() {
		if (isMac()) return OS.osx;
		if (isWindows()) return OS.windows;
		return OS.linux;
	}
}
