package io.intino.consul.framework;

import io.intino.consul.terminal.Terminal;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Type;
import java.nio.file.CopyOption;
import java.nio.file.StandardOpenOption;
import java.util.List;
import java.util.Map;
import java.util.stream.Stream;

public interface Activity {
	String id();

	Result configure(Context context, Store store);

	Map<String, String> currentConfiguration();

	Result onStart();

	Result onStop();

	record Context(Terminal terminal, System system, String hostName, String observer, String ss,
				   Map<String, String> initialConfiguration) {
	}

	record Result(boolean success, String remarks) {
	}

	interface System {
		String name();

		OperatingSystem operatingSystem() throws Exception;

		HDD hdd();

		RAM ram();

		CPU cpu();

		Network network();

		record HDD(long capacityMB, long usedMB, long maxOpenFiles) {
		}

		record RAM(long capacityMB, long usedMB) {
		}

		record CPU(int cores, long usagePercent, double temperature) {
		}

		record Network(String localIP, String publicIP, String ISP) {
		}


		interface OperatingSystem {
			enum Name {Windows, Mac, Unix}

			String name();

			String fullName();

			long bootTimeSeconds();

			long userHz();

			long pageSize();

			Measurements measurements();

			FileSystem fileSystem() throws IOException;

			ProcessRunner processRunner() throws Exception;

			int createConnection(int port);

			void removeConnection(int port);

			List<? extends OSProcess> processes() throws Exception;

			int processCount();
		}

		interface Measurements {
			long usageRAM();

			long usageHDD();

			double usageCPU();

			double usageSystem();

			int usageFiles();

			long usageThreads();

			long dataReceived();

			long dataSent();

			double temperatureKernel();

			double temperatureExternal();
		}

		interface OSProcess {
			String name();

			String path();

			String commandLine();

			List<String> arguments();

			String user();

			String group();

			State state();

			boolean isOSProcess();

			boolean isSystemService();

			String systemServiceName();

			int processID();

			int parentProcessID();

			int threadCount();

			long virtualSize();

			long residentSetSize();

			long kernelTime();

			long userTime();

			long upTime();

			long startTime();

			long bytesRead();

			long bytesWritten();

			long openFiles();

			long openFilesLimit();

			double processCpuLoadCumulative();

			int getBitness();

			enum State {
				NEW,
				RUNNING,
				SLEEPING,
				WAITING,
				ZOMBIE,
				STOPPED,
				OTHER,
				INVALID,
				SUSPENDED
			}
		}

		interface ProcessRunner {
			default String execute(String... command) throws Exception {
				return execute(List.of(command), 3 * 60);
			}

			default String execute(List<String> command) throws Exception {
				return execute(command, 3 * 60);
			}

			String execute(List<String> command, int timeoutSeconds) throws Exception;
		}

		interface FileSystem {
			boolean exists(String path);

			String readFile(String path);

			byte[] readFileBytes(String path);

			InputStream openFile(String path);

			void writeString(String path, String content);

			void writeString(String path, String content, StandardOpenOption... options);

			void changeOwner(String path, String user, String group);

			void copy(String source, String target, CopyOption... options);

			void mkdirs(String path);

			void delete(String path);

			void deleteDirectory(String path);

			List<String> listDirectory(String path);
		}

	}

	interface Store {
		String Separator = "#";

		void put(String key, Object object);

		<T> T get(String key, Class<T> type);

		<T> T get(String key, Type typeOfT);

		Stream<String> keys();

		File workingDirectory();

		void remove(String key);
	}
}
