package io.intino.consul.container.box.os.remote;

import com.jcraft.jsch.ChannelSftp;
import com.jcraft.jsch.JSchException;
import com.jcraft.jsch.Session;
import com.jcraft.jsch.SftpException;
import io.intino.alexandria.logger.Logger;
import io.intino.consul.framework.Activity;

import java.io.*;
import java.nio.file.CopyOption;
import java.nio.file.StandardOpenOption;
import java.util.Arrays;
import java.util.List;

import static java.nio.file.StandardOpenOption.APPEND;
import static java.nio.file.StandardOpenOption.CREATE;

public class RemoteFileSystem implements Activity.System.FileSystem, Closeable {
	private final Session session;

	public RemoteFileSystem(Session session) {
		this.session = session;
	}

	@Override
	public boolean exists(String path) {
		try {
			String executed = new RemoteProcessRunner(session).execute("ls", path).trim();
			return executed.startsWith(path);
		} catch (Exception e) {
			Logger.error(path + ": " + e.getMessage());
			return false;
		}
	}

	@Override
	public String readFile(String path) {
		try {
			return read(path).toString();
		} catch (SftpException | JSchException e) {
			if (e.getMessage().contains("No such file")) return null;
			Logger.error(path + ": " + e.getMessage());
			return null;
		}
	}

	@Override
	public byte[] readFileBytes(String path) {
		try {
			return read(path).toByteArray();
		} catch (JSchException | SftpException e) {
			Logger.error(path + ": " + e.getMessage());
			return null;
		}
	}

	@Override
	public InputStream openFile(String path) {
		ChannelSftp sftp;
		try {
			sftp = openSftp();
			return new SftpInputStream(sftp.get(path), sftp);
		} catch (JSchException | SftpException e) {
			Logger.error(e);
			return InputStream.nullInputStream();
		}
	}

	private ByteArrayOutputStream read(String path) throws JSchException, SftpException {
		ChannelSftp sftp = null;
		try {
			sftp = openSftp();
			ByteArrayOutputStream stream = new ByteArrayOutputStream();
			sftp.get(path, stream);
			sftp.disconnect();
			return stream;
		} finally {
			if (sftp != null) sftp.disconnect();
		}
	}

	private ByteArrayOutputStream open(String path) throws JSchException, SftpException {
		ChannelSftp sftp = null;
		try {
			sftp = openSftp();
			ByteArrayOutputStream stream = new ByteArrayOutputStream();
			sftp.get(path, stream);
			sftp.disconnect();
			return stream;
		} finally {
			if (sftp != null) sftp.disconnect();
		}
	}

	@Override
	public void writeString(String path, String content) {
		writeString(path, content, CREATE);
	}

	@Override
	public void writeString(String path, String content, StandardOpenOption... options) {
		mkdirs(new File(path).getParent());
		ChannelSftp sftp = null;
		try {
			sftp = openSftp();
			OutputStream outputStream = sftp.put(path, Arrays.asList(options).contains(APPEND) ? ChannelSftp.APPEND : ChannelSftp.OVERWRITE);
			outputStream.write(content.getBytes());
			outputStream.close();
		} catch (SftpException | IOException | JSchException e) {
			Logger.error(e.getMessage() + ". " + path);
		} finally {
			if (sftp != null) sftp.disconnect();
		}
	}

	@Override
	public void changeOwner(String path, String user, String group) {
		try {
			new RemoteProcessRunner(session).execute("chown", "-R", user + ":" + group, path);
		} catch (Exception e) {
			Logger.error(e.getMessage() + ". " + path);
		}
	}

	@Override
	public void changePermissions(String path, String mask) {
		try {
			new RemoteProcessRunner(session).execute("chmod", "-R", mask, path);
		} catch (Exception e) {
			Logger.error(e.getMessage() + ". " + path);
		}
	}

	@Override
	public void copy(String source, String target, CopyOption... options) {
		ChannelSftp sftp = null;
		try {
			sftp = openSftp();
			sftp.put(source, target);
		} catch (SftpException | JSchException e) {
			Logger.error(e.getMessage() + ". source: " + source + "; target: " + target);
		} finally {
			if (sftp != null) sftp.disconnect();
		}
	}

	@Override
	public void mkdirs(String path) {
		try {
			new RemoteProcessRunner(session).execute("mkdir", "-p", path);
		} catch (Exception e) {
			Logger.error(e.getMessage() + ". " + path);
		}
	}

	@Override
	public void delete(String path) {
		try {
			new RemoteProcessRunner(session).execute("rm -f", path);
		} catch (Exception e) {
			Logger.error(path + ": " + e.getMessage());
		}
	}

	@Override
	public void deleteDirectory(String path) {
		try {
			new RemoteProcessRunner(session).execute("rm", "-Rf", path);
		} catch (Exception e) {
			Logger.error(path + ": " + e.getMessage());
		}
	}

	@Override
	public List<String> listDirectory(String path) {
		try {
			String response = new RemoteProcessRunner(session).execute("ls", path).trim();
			return response.lines().map(String::trim).filter(l -> !l.isEmpty() && !l.isBlank()).toList();
		} catch (Exception e) {
			Logger.error(path + ": " + e.getMessage());
			return List.of();
		}

	}

	private ChannelSftp openSftp() throws JSchException {
		ChannelSftp channel = (ChannelSftp) session.openChannel("sftp");
		channel.connect(60_000);
		return channel;
	}

	@Override
	public void close() {
	}
}
