/*
 * Decompiled with CFR 0.152.
 */
package io.intino.alexandria.terminal;

import io.intino.alexandria.event.Event;
import io.intino.alexandria.jms.BrokerConnector;
import io.intino.alexandria.jms.ConnectionConfig;
import io.intino.alexandria.jms.ConnectionListener;
import io.intino.alexandria.jms.DurableTopicConsumer;
import io.intino.alexandria.jms.JmsConsumer;
import io.intino.alexandria.jms.JmsProducer;
import io.intino.alexandria.jms.MessageReader;
import io.intino.alexandria.jms.MessageWriter;
import io.intino.alexandria.jms.QueueConsumer;
import io.intino.alexandria.jms.QueueProducer;
import io.intino.alexandria.jms.TopicConsumer;
import io.intino.alexandria.jms.TopicProducer;
import io.intino.alexandria.logger.Logger;
import io.intino.alexandria.terminal.Broker;
import io.intino.alexandria.terminal.Connector;
import io.intino.alexandria.terminal.EventOutBox;
import io.intino.alexandria.terminal.MessageOutBox;
import io.intino.alexandria.terminal.MessageTranslator;
import java.io.File;
import java.io.IOException;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import javax.jms.Connection;
import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.MessageConsumer;
import javax.jms.Session;
import javax.jms.TemporaryQueue;
import org.apache.activemq.ActiveMQConnection;
import org.apache.activemq.ActiveMQSession;
import org.apache.activemq.command.ActiveMQDestination;

public class JmsConnector
implements Connector {
    private final Map<String, JmsProducer> producers;
    private final Map<String, JmsConsumer> consumers;
    private final Map<String, List<Consumer<Event>>> eventConsumers;
    private final Map<String, List<Connector.MessageConsumer>> messageConsumers;
    private final Map<Consumer<Event>, Integer> jmsEventConsumers;
    private final Map<Connector.MessageConsumer, Integer> jmsMessageConsumers;
    private final ConnectionConfig config;
    private final boolean transactedSession;
    private final AtomicBoolean connected = new AtomicBoolean(false);
    private final AtomicBoolean started = new AtomicBoolean(false);
    private EventOutBox eventOutBox;
    private MessageOutBox messageOutBox;
    private Connection connection;
    private Session session;
    private ScheduledExecutorService scheduler;
    private final ExecutorService eventDispatcher;

    public JmsConnector(ConnectionConfig config, File outboxDirectory) {
        this(config, false, outboxDirectory);
    }

    public JmsConnector(ConnectionConfig config, boolean transactedSession, File outBoxDirectory) {
        this.config = config;
        this.transactedSession = transactedSession;
        this.producers = new HashMap<String, JmsProducer>();
        this.consumers = new HashMap<String, JmsConsumer>();
        this.jmsEventConsumers = new HashMap<Consumer<Event>, Integer>();
        this.jmsMessageConsumers = new HashMap<Connector.MessageConsumer, Integer>();
        this.eventConsumers = new HashMap<String, List<Consumer<Event>>>();
        this.messageConsumers = new HashMap<String, List<Connector.MessageConsumer>>();
        if (outBoxDirectory != null) {
            this.eventOutBox = new EventOutBox(new File(outBoxDirectory, "events"));
            this.messageOutBox = new MessageOutBox(new File(outBoxDirectory, "requests"));
        }
        this.eventDispatcher = Executors.newSingleThreadExecutor(new NamedThreadFactory("jms-connector"));
    }

    @Override
    public String clientId() {
        return this.config.clientId();
    }

    public void start() {
        if (this.config.url() == null || this.config.url().isEmpty()) {
            Logger.warn("Invalid broker URL (" + this.config.url() + "). Connection aborted");
            return;
        }
        try {
            this.connect();
        }
        catch (JMSException e) {
            Logger.error(e);
        }
        this.started.set(true);
        if (this.scheduler == null) {
            this.scheduler = Executors.newScheduledThreadPool(1);
            this.scheduler.scheduleAtFixedRate(this::checkConnection, 15L, 10L, TimeUnit.MINUTES);
        }
    }

    private void connect() throws JMSException {
        if (!Broker.isRunning(this.config.url())) {
            Logger.warn("Broker (" + this.config.url() + ") unreachable .Connection aborted");
            return;
        }
        this.initConnection();
        if (this.connection != null && ((ActiveMQConnection)this.connection).isStarted()) {
            this.clearProducers();
            this.session = this.createSession(this.transactedSession);
            if (this.session != null && ((ActiveMQSession)this.session).isRunning()) {
                this.connected.set(true);
                this.recoverEventsAndMessages();
            }
        }
    }

    @Override
    public synchronized void sendEvent(String path, Event event) {
        ArrayList consumers = new ArrayList(this.eventConsumers.getOrDefault(path, Collections.emptyList()));
        for (Consumer c : consumers) {
            c.accept(event);
        }
        this.eventDispatcher.execute(() -> {
            if (!this.doSendEvent(path, event) && this.eventOutBox != null) {
                this.eventOutBox.push(path, event);
            }
        });
    }

    @Override
    public synchronized void sendEvents(String path, List<Event> events) {
        ArrayList<Consumer> consumers = new ArrayList<Consumer>(this.eventConsumers.getOrDefault(path, Collections.emptyList()));
        consumers.forEach(events::forEach);
        this.eventDispatcher.execute(() -> {
            if (!this.doSendEvents(path, events) && this.eventOutBox != null) {
                events.forEach(e -> this.eventOutBox.push(path, (Event)e));
            }
        });
    }

    @Override
    public synchronized void sendEvents(String path, List<Event> events, int expirationInSeconds) {
        ArrayList<Consumer> consumers = new ArrayList<Consumer>(this.eventConsumers.getOrDefault(path, Collections.emptyList()));
        consumers.forEach(events::forEach);
        this.eventDispatcher.execute(() -> {
            if (!this.doSendEvents(path, events, expirationInSeconds) && this.eventOutBox != null) {
                events.forEach(e -> this.eventOutBox.push(path, (Event)e));
            }
        });
    }

    @Override
    public synchronized void sendEvent(String path, Event event, int expirationInSeconds) {
        ArrayList consumers = new ArrayList(this.eventConsumers.getOrDefault(path, Collections.emptyList()));
        for (Consumer eventConsumer : consumers) {
            eventConsumer.accept(event);
        }
        this.eventDispatcher.execute(() -> {
            if (!this.doSendEvent(path, event, expirationInSeconds) && this.eventOutBox != null) {
                this.eventOutBox.push(path, event);
            }
        });
    }

    @Override
    public void attachListener(String path, Consumer<Event> onEventReceived) {
        this.registerEventConsumer(path, null, onEventReceived);
        this.attach(path, onEventReceived);
    }

    @Override
    public void attachListener(String path, Consumer<Event> onEventReceived, String messageSelector) {
        this.registerEventConsumer(path, messageSelector, onEventReceived);
        this.attach(path, onEventReceived);
    }

    @Override
    public void sendQueueMessage(String path, String message) {
        this.recoverEventsAndMessages();
        if (!this.doSendMessageToQueue(path, message) && this.messageOutBox != null) {
            this.messageOutBox.push(path, message);
        }
    }

    @Override
    public void sendTopicMessage(String path, String message) {
        this.recoverEventsAndMessages();
        if (!this.doSendMessageToTopic(path, message) && this.messageOutBox != null) {
            this.messageOutBox.push(path, message);
        }
    }

    @Override
    public void attachListener(String path, String subscriberId, Consumer<Event> onEventReceived) {
        this.registerEventConsumer(path, subscriberId, null, onEventReceived);
        this.attach(path, onEventReceived);
    }

    @Override
    public void attachListener(String path, String subscriberId, Consumer<Event> onEventReceived, Predicate<Instant> filter, String messageSelector) {
        if (filter == null) {
            this.attachListener(path, subscriberId, onEventReceived, messageSelector);
        }
        this.registerEventConsumer(path, subscriberId, messageSelector, onEventReceived);
        JmsConsumer consumer = this.consumers.get(path);
        if (consumer == null) {
            return;
        }
        Consumer<Message> eventConsumer = m -> {
            Instant timestamp = this.timestamp((Message)m);
            if (timestamp != null && filter != null && filter.test(timestamp)) {
                MessageTranslator.deserialize(m).forEach(onEventReceived);
            }
        };
        this.jmsEventConsumers.put(onEventReceived, eventConsumer.hashCode());
        consumer.listen(eventConsumer);
    }

    @Override
    public void attachListener(String path, String subscriberId, Consumer<Event> onEventReceived, String messageSelector) {
        this.registerEventConsumer(path, subscriberId, messageSelector, onEventReceived);
        this.attach(path, onEventReceived);
    }

    @Override
    public void attachListener(String path, Connector.MessageConsumer onMessageReceived) {
        this.registerMessageConsumer(path, onMessageReceived);
        this.attach(path, onMessageReceived);
    }

    @Override
    public void attachListener(String path, String subscriberId, Connector.MessageConsumer onMessageReceived) {
        this.registerMessageConsumer(path, subscriberId, onMessageReceived);
        this.attach(path, onMessageReceived);
    }

    private void attach(String path, Connector.MessageConsumer onMessageReceived) {
        JmsConsumer consumer = this.consumers.get(path);
        if (consumer == null) {
            return;
        }
        Consumer<Message> messageConsumer = m -> onMessageReceived.accept(MessageReader.textFrom(m), this.callback((Message)m));
        this.jmsMessageConsumers.put(onMessageReceived, messageConsumer.hashCode());
        consumer.listen(messageConsumer);
    }

    private void attach(String path, Consumer<Event> onEventReceived) {
        JmsConsumer consumer = this.consumers.get(path);
        if (consumer == null) {
            return;
        }
        Consumer<Message> eventConsumer = m -> MessageTranslator.deserialize(m).forEach(onEventReceived);
        this.jmsEventConsumers.put(onEventReceived, eventConsumer.hashCode());
        consumer.listen(eventConsumer);
    }

    private Instant timestamp(Message m) {
        try {
            return Instant.ofEpochMilli(m.getJMSTimestamp());
        }
        catch (JMSException e) {
            Logger.error(e);
            return null;
        }
    }

    @Override
    public void detachListeners(Consumer<Event> consumer) {
        this.eventConsumers.values().forEach(list -> list.remove(consumer));
        this.detach(this.jmsEventConsumers.get(consumer));
    }

    @Override
    public void detachListeners(Connector.MessageConsumer consumer) {
        this.messageConsumers.values().forEach(list -> list.remove(consumer));
        this.detach(this.jmsMessageConsumers.get(consumer));
    }

    private void detach(Integer consumerCode) {
        if (consumerCode == null) {
            return;
        }
        for (JmsConsumer jc : this.consumers.values()) {
            List<Consumer> toRemove = jc.listeners().stream().filter(l -> l.hashCode() == consumerCode.intValue()).collect(Collectors.toList());
            toRemove.forEach(jc::removeListener);
        }
    }

    @Override
    public void createSubscription(String path, String subscriberId) {
        if (this.session != null && !this.consumers.containsKey(path)) {
            this.consumers.put(path, this.durableTopicConsumer(path, subscriberId, null));
        }
    }

    @Override
    public void destroySubscription(String subscriberId) {
        try {
            this.session.unsubscribe(subscriberId);
        }
        catch (JMSException e) {
            Logger.error(e);
        }
    }

    @Override
    public void detachListeners(String path) {
        if (this.consumers.containsKey(path)) {
            this.consumers.get(path).close();
            this.consumers.remove(path);
            this.eventConsumers.get(path).clear();
            this.messageConsumers.get(path).clear();
        }
    }

    @Override
    public Message requestResponse(String path, Message message) {
        return this.requestResponse(path, message, this.config.defaultTimeoutAmount(), this.config.defaultTimeoutUnit());
    }

    @Override
    public Message requestResponse(String path, Message message, long timeout, TimeUnit timeUnit) {
        if (this.session == null) {
            Logger.error("Connection lost. Invalid session");
            return null;
        }
        try {
            QueueProducer producer = new QueueProducer(this.session, path);
            TemporaryQueue temporaryQueue = this.session.createTemporaryQueue();
            message.setJMSReplyTo(temporaryQueue);
            message.setJMSCorrelationID(JmsConnector.createRandomString());
            MessageConsumer consumer = this.session.createConsumer(temporaryQueue);
            CompletableFuture<Message> future = new CompletableFuture<Message>();
            consumer.setMessageListener(future::complete);
            this.sendMessage(producer, message, 100);
            producer.close();
            Message response = JmsConnector.waitFor(future, timeout, timeUnit);
            consumer.close();
            return response;
        }
        catch (InterruptedException | ExecutionException | TimeoutException | JMSException e) {
            Logger.error(e.getMessage());
            return null;
        }
    }

    private static Message waitFor(Future<Message> future, long timeout, TimeUnit timeUnit) throws ExecutionException, InterruptedException, TimeoutException {
        return timeout <= 0L || timeUnit == null ? future.get() : future.get(timeout, timeUnit);
    }

    @Override
    public void requestResponse(String path, Message message, String responsePath) {
        try {
            message.setJMSReplyTo(this.session.createQueue(responsePath));
            message.setJMSCorrelationID(JmsConnector.createRandomString());
            this.sendMessage(this.producers.get(path), message);
        }
        catch (JMSException e) {
            Logger.error(e);
        }
    }

    @Override
    public long defaultTimeoutAmount() {
        return this.config.defaultTimeoutAmount();
    }

    @Override
    public TimeUnit defaultTimeoutUnit() {
        return this.config.defaultTimeoutUnit();
    }

    public Connection connection() {
        return this.connection;
    }

    public Session session() {
        return this.session;
    }

    public void stop() {
        try {
            this.consumers.values().forEach(JmsConsumer::close);
            this.consumers.clear();
            this.producers.values().forEach(JmsProducer::close);
            this.producers.clear();
            if (this.session != null) {
                this.session.close();
            }
            if (this.connection != null) {
                this.connection.close();
            }
            this.session = null;
            this.connection = null;
        }
        catch (Exception e) {
            Logger.error(e);
        }
    }

    private Session createSession(boolean transactedSession) throws JMSException {
        return this.connection.createSession(transactedSession, transactedSession ? 0 : 1);
    }

    private void registerEventConsumer(String path, String messageSelector, Consumer<Event> onEventReceived) {
        this.eventConsumers.putIfAbsent(path, new CopyOnWriteArrayList());
        this.eventConsumers.get(path).add(onEventReceived);
        if (this.session != null && !this.consumers.containsKey(path)) {
            this.consumers.put(path, this.topicConsumer(path, messageSelector));
        }
    }

    private void registerEventConsumer(String path, String subscriberId, String messageSelector, Consumer<Event> onEventReceived) {
        this.eventConsumers.putIfAbsent(path, new CopyOnWriteArrayList());
        this.eventConsumers.get(path).add(onEventReceived);
        if (this.session != null && !this.consumers.containsKey(path)) {
            this.consumers.put(path, this.durableTopicConsumer(path, subscriberId, messageSelector));
        }
    }

    private void registerMessageConsumer(String path, Connector.MessageConsumer onMessageReceived) {
        this.messageConsumers.putIfAbsent(path, new CopyOnWriteArrayList());
        this.messageConsumers.get(path).add(onMessageReceived);
        if (this.session != null && !this.consumers.containsKey(path)) {
            this.consumers.put(path, this.queueConsumer(path));
        }
    }

    private void registerMessageConsumer(String path, String subscriberId, Connector.MessageConsumer onMessageReceived) {
        this.messageConsumers.putIfAbsent(path, new CopyOnWriteArrayList());
        this.messageConsumers.get(path).add(onMessageReceived);
        if (this.session != null && !this.consumers.containsKey(path)) {
            this.consumers.put(path, subscriberId == null ? this.topicConsumer(path, null) : this.durableTopicConsumer(path, subscriberId, null));
        }
    }

    private boolean doSendEvent(String path, Event event) {
        return this.doSendEvent(path, event, 0);
    }

    private boolean doSendEvents(String path, List<Event> event) {
        return this.doSendEvents(path, event, 0);
    }

    private boolean doSendEvent(String path, Event event, int expirationTimeInSeconds) {
        if (this.cannotSendMessage()) {
            return false;
        }
        try {
            return this.sendMessage(this.topicProducer(path), MessageTranslator.serialize(event), expirationTimeInSeconds);
        }
        catch (IOException | JMSException e) {
            Logger.error(e);
            return false;
        }
    }

    private boolean doSendEvents(String path, List<Event> events, int expirationTimeInSeconds) {
        if (this.cannotSendMessage()) {
            return false;
        }
        try {
            return this.sendMessage(this.topicProducer(path), MessageTranslator.serialize(events), expirationTimeInSeconds);
        }
        catch (IOException | JMSException e) {
            Logger.error(e);
            return false;
        }
    }

    private boolean doSendMessageToQueue(String path, String message) {
        try {
            if (this.cannotSendMessage()) {
                return false;
            }
            return this.sendMessage(this.queueProducer(path), MessageWriter.write(message));
        }
        catch (JMSException e) {
            Logger.error(e);
            return false;
        }
    }

    private boolean doSendMessageToTopic(String path, String message) {
        try {
            if (this.cannotSendMessage()) {
                return false;
            }
            return this.sendMessage(this.queueProducer(path), MessageWriter.write(message));
        }
        catch (JMSException e) {
            Logger.error(e);
            return false;
        }
    }

    private JmsProducer topicProducer(String path) throws JMSException {
        if (!this.producers.containsKey(path)) {
            this.producers.put(path, new TopicProducer(this.session, path));
        }
        return this.producers.get(path);
    }

    private JmsProducer queueProducer(String path) throws JMSException {
        if (!this.producers.containsKey(path)) {
            this.producers.put(path, new QueueProducer(this.session, path));
        }
        return this.producers.get(path);
    }

    private boolean cannotSendMessage() {
        return this.session == null || !this.connected.get();
    }

    private boolean sendMessage(JmsProducer producer, Message message) {
        return this.sendMessage(producer, message, 0);
    }

    private boolean sendMessage(JmsProducer producer, Message message, int expirationTimeInSeconds) {
        boolean[] result = new boolean[]{false};
        try {
            Thread thread = new Thread(() -> {
                result[0] = producer.produce(message, expirationTimeInSeconds);
            });
            thread.start();
            thread.join(1000L);
            thread.interrupt();
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
        return result[0];
    }

    private ConnectionListener connectionListener() {
        return new ConnectionListener(){

            @Override
            public void transportInterupted() {
                Logger.warn("Connection with Data Hub (" + JmsConnector.this.config.url() + ") interrupted!");
                JmsConnector.this.connected.set(false);
            }

            @Override
            public void transportResumed() {
                Logger.info("Connection with Data Hub (" + JmsConnector.this.config.url() + ") established!");
                JmsConnector.this.connected.set(true);
                JmsConnector.this.recoverConsumers();
            }
        };
    }

    private void clearProducers() {
        this.producers.values().forEach(JmsProducer::close);
        this.producers.clear();
    }

    private void clearConsumers() {
        this.consumers.values().forEach(JmsConsumer::close);
        this.consumers.clear();
    }

    private JmsConsumer topicConsumer(String path, String messageSelector) {
        try {
            return new TopicConsumer(this.session, path, messageSelector);
        }
        catch (JMSException e) {
            Logger.error(e);
            return null;
        }
    }

    private JmsConsumer durableTopicConsumer(String path, String subscriberId, String messageSelector) {
        try {
            return new DurableTopicConsumer(this.session, path, messageSelector, subscriberId);
        }
        catch (JMSException e) {
            Logger.error(e);
            return null;
        }
    }

    private QueueConsumer queueConsumer(String path) {
        try {
            return new QueueConsumer(this.session, path);
        }
        catch (JMSException e) {
            Logger.error(e);
            return null;
        }
    }

    private void recoverConsumers() {
        if (!this.started.get()) {
            return;
        }
        if (!this.eventConsumers.isEmpty() && this.consumers.isEmpty()) {
            for (String path : this.eventConsumers.keySet()) {
                this.consumers.put(path, this.topicConsumer(path, path));
            }
        }
        if (!this.messageConsumers.isEmpty() && this.consumers.isEmpty()) {
            for (String path : this.messageConsumers.keySet()) {
                if (!this.consumers.containsKey(path) && this.session != null) {
                    this.consumers.put(path, this.queueConsumer(path));
                }
                for (Connector.MessageConsumer mConsumer : this.messageConsumers.get(path)) {
                    Consumer<Message> messageConsumer = m -> mConsumer.accept(MessageReader.textFrom(m), this.callback((Message)m));
                    this.jmsMessageConsumers.put(mConsumer, messageConsumer.hashCode());
                    this.consumers.get(path).listen(messageConsumer);
                }
            }
        }
    }

    private synchronized void recoverEventsAndMessages() {
        this.recoverEvents();
        this.recoverMessages();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void recoverEvents() {
        if (this.eventOutBox == null) {
            return;
        }
        EventOutBox eventOutBox = this.eventOutBox;
        synchronized (eventOutBox) {
            if (this.eventOutBox.isEmpty()) {
                return;
            }
            Logger.info("Recovering events...");
            while (!this.eventOutBox.isEmpty()) {
                for (Map.Entry<String, Event> event : this.eventOutBox.get()) {
                    if (this.doSendEvent(event.getKey(), event.getValue())) {
                        this.eventOutBox.pop();
                        continue;
                    }
                    return;
                }
            }
        }
        Logger.info("All events recovered!");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void recoverMessages() {
        if (this.messageOutBox == null) {
            return;
        }
        MessageOutBox messageOutBox = this.messageOutBox;
        synchronized (messageOutBox) {
            if (!this.messageOutBox.isEmpty()) {
                while (!this.messageOutBox.isEmpty()) {
                    Map.Entry<String, String> message = this.messageOutBox.get();
                    if (message == null) continue;
                    if (!this.doSendMessageToQueue(message.getKey(), message.getValue())) break;
                    this.messageOutBox.pop();
                }
            }
        }
    }

    private void checkConnection() {
        if (this.session != null && this.config.url().startsWith("failover") && !this.connected.get()) {
            Logger.debug("Data-hub currently disconnected. Waiting for reconnection...");
            return;
        }
        if (this.connection != null && ((ActiveMQConnection)this.connection).isStarted() && this.session != null && ((ActiveMQSession)this.session).isRunning()) {
            this.connected.set(true);
            return;
        }
        Logger.debug("Restarting data-hub connection...");
        this.stop();
        try {
            this.connect();
        }
        catch (JMSException jMSException) {
            // empty catch block
        }
        this.connected.set(true);
    }

    private void initConnection() {
        try {
            this.connection = BrokerConnector.createConnection(this.config, this.connectionListener());
            if (this.connection != null) {
                if (this.config.clientId() != null && !this.config.clientId().isEmpty()) {
                    this.connection.setClientID(this.config.clientId());
                }
                this.connection.start();
            }
        }
        catch (JMSException e) {
            Logger.error(e);
        }
    }

    private String callback(Message m) {
        try {
            ActiveMQDestination replyTo = (ActiveMQDestination)m.getJMSReplyTo();
            return replyTo == null ? null : replyTo.getPhysicalName();
        }
        catch (JMSException e) {
            return null;
        }
    }

    public static String createRandomString() {
        Random random = new Random(System.currentTimeMillis());
        long randomLong = random.nextLong();
        return Long.toHexString(randomLong);
    }

    public static class NamedThreadFactory
    implements ThreadFactory {
        private final AtomicInteger sequence = new AtomicInteger(1);
        private final String prefix;

        public NamedThreadFactory(String prefix) {
            this.prefix = prefix;
        }

        @Override
        public Thread newThread(Runnable r) {
            Thread thread = new Thread(r);
            int seq = this.sequence.getAndIncrement();
            thread.setName(this.prefix + (String)(seq > 1 ? "-" + seq : ""));
            if (!thread.isDaemon()) {
                thread.setDaemon(true);
            }
            return thread;
        }
    }
}

