package io.intino.consul.container.box.service.requests;

import io.intino.alexandria.logger.Logger;
import io.intino.consul.container.box.ContainerBox;
import io.intino.consul.container.box.ContainerConfiguration;
import io.intino.consul.container.box.Utils;
import io.intino.consul.container.box.actions.WindowsUpgradeCommand;

import javax.jms.BytesMessage;
import javax.jms.JMSException;
import javax.jms.Message;
import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;

import static io.intino.alexandria.Base64.decode;

public class UpgradeRequest extends ContainerRequest {
	public static final String ID = "upgrade";
	private final ContainerBox box;
	private final ContainerConfiguration configuration;
	private static final AtomicBoolean upgrading = new AtomicBoolean(false);

	public UpgradeRequest(ContainerBox box, ContainerConfiguration configuration) {
		this.box = box;
		this.configuration = configuration;
	}

	@Override
	public String id() {
		return ID;
	}

	@Override
	public RequestResult responseTo(Message request) {
		if (upgrading.get()) return new RequestResult(false, "Upgrade in progress");
		upgrading.set(true);
		BytesMessage message = (BytesMessage) request;
		try {
			String version = parameter(request, "version");
			Logger.info("Upgrading Consul Container to: " + version);
			File installDirectory = installDirectory();
			File jarFile = jarFile();
			String jarName = jarFile.getName().toLowerCase().replace(".jar", "");
			saveConfigurationFiles(request, installDirectory, jarName);
			byte[] payload = new byte[(int) message.getBodyLength()];
			message.readBytes(payload);
			File tempFile = new File(installDirectory, "consul.new.jar");
			Files.write(tempFile.toPath(), payload);
			if (Utils.isUnix()) {
				new Thread(() -> {
					try {
						box.beforeStop();
						Files.move(tempFile.toPath(), jarFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
						runProcess(new ProcessBuilder().command("systemctl", "restart", jarName));
					} catch (IOException e) {
						Logger.error(e);
					}
				}).start();
			} else if (Utils.isWindows()) new WindowsUpgradeCommand(installDirectory).execute();
			upgrading.set(false);
			return new RequestResult(true, "Consul upgraded. Restart in progress...");
		} catch (Throwable e) {
			Logger.error(e);
			return new RequestResult(false, e.getMessage());
		} finally {
			upgrading.set(false);
		}
	}

	@Override
	public boolean isAvailable(Message message) {
		return true;
	}


	private void saveConfigurationFiles(Message request, File installDirectory, String serviceName) throws JMSException, IOException, InterruptedException {
		String jts = parameter(request, "jts");
		String jks = parameter(request, "jks");
		if (jts == null || jks == null) return;
		String password = parameter(request, "password");
		Files.write(configuration.keyStoreFile().toPath(), decode(jks));
		Files.write(configuration.trustStoreFile().toPath(), decode(jts));
		if (Utils.isUnix()) updateServiceFile(password, serviceName);
		else if (Utils.isWindows()) updateWrapperFile(installDirectory, password);
	}

	private void updateWrapperFile(File installDirectory, String password) {
		try {
			File file = new File(installDirectory, "win64" + File.separator + "wrapper.conf");
			String content = Files.readString(file.toPath());
			Files.writeString(file.toPath(), content.replaceAll("store_password=.*\"", "store_password=" + password + "\""));
		} catch (IOException e) {
			Logger.error(e);
		}
	}

	private static void updateServiceFile(String password, String serviceName) throws IOException, InterruptedException {
		File file = serviceFile(serviceName);
		if (file != null && file.exists()) {
			String content = Files.readString(file.toPath());
			content = content.replaceAll("store_password=.* ", "store_password=" + password + " ");
			Files.writeString(file.toPath(), content);
			Process process = new ProcessBuilder("systemctl", "daemon-reload").redirectErrorStream(true).start();
			process.waitFor(4, TimeUnit.SECONDS);
		}
	}

	private static void runProcess(ProcessBuilder processBuilder) {
		new Thread(() -> {
			try {
				Thread.sleep(3000);
				processBuilder.start();
			} catch (InterruptedException | IOException e) {
				Logger.error(e);
			}
		}).start();
	}


	private static File serviceFile(String serviceName) throws IOException {
		try {
			Process process = new ProcessBuilder("systemctl", "status", serviceName + ".service").redirectErrorStream(true).start();
			process.waitFor(4, TimeUnit.SECONDS);
			String result = new String(process.getInputStream().readAllBytes());
			String loaded = result.lines().map(String::trim).filter(l -> l.startsWith("Loaded:")).findFirst().orElse(null);
			if (loaded == null) return null;
			return new File(loaded.substring(loaded.indexOf("(") + 1, loaded.indexOf(";")));
		} catch (InterruptedException e) {
			Logger.error(e);
			return null;
		}
	}

	public static File installDirectory() {
		return new File(UpgradeRequest.class.getProtectionDomain().getCodeSource().getLocation().getFile().replace("%20", " ")).getParentFile();
	}

	private File jarFile() {
		return new File(this.getClass().getProtectionDomain().getCodeSource().getLocation().getFile().replace("%20", " "));
	}


}
