package io.intino.consul.container.box.service.requests;

import io.intino.alexandria.logger.Logger;

import javax.jms.JMSException;
import javax.jms.Message;
import java.io.File;
import java.io.IOException;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Set;
import java.util.stream.Stream;

import static java.io.File.listRoots;
import static java.util.stream.Collectors.toList;

public class ExploreFileSystemRequest extends ContainerRequest {
	public static final String ID = "exploreFileSystem";

	@Override
	public String id() {
		return ID;
	}

	@Override
	public RequestResult responseTo(Message request) {
		try {
			String root = parameter(request, "path");
			List<File> roots = Arrays.asList(listRoots());
			if (root.isEmpty())
				return new RequestResult(true, roots.stream().map(File::getAbsolutePath).collect(toList()));
			return new RequestResult(true, walk(Path.of(root), roots.stream().anyMatch(r -> r.getAbsolutePath().equals(root)) ? 1 : 4)
					.map(p -> p.toAbsolutePath() + (p.toFile().isDirectory() ? File.separator : ""))
					.collect(toList()));
		} catch (JMSException e) {
			Logger.error(e);
			return new RequestResult(false, e.getMessage());
		}
	}

	@Override
	public boolean isAvailable(Message message) {
		return true;
	}

	private Stream<Path> walk(Path path, int depth) {
		try {
			List<Path> paths = new ArrayList<>();
			Files.walkFileTree(
					path,
					Set.of(),
					depth, new SimpleFileVisitor<>() {
						@Override
						public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) {
							paths.add(file);
							return FileVisitResult.CONTINUE;
						}

						@Override
						public FileVisitResult visitFileFailed(Path file, IOException e) {
							return FileVisitResult.SKIP_SUBTREE;
						}

						@Override
						public FileVisitResult preVisitDirectory(Path dir, BasicFileAttributes attrs) {
							return FileVisitResult.CONTINUE;
						}

						@Override
						public FileVisitResult postVisitDirectory(Path dir, IOException exc) {
							paths.add(dir);
							return FileVisitResult.CONTINUE;

						}
					});
			return paths.stream();
		} catch (IOException e) {
			Logger.error(e);
			return Stream.of();
		}
	}
}