package io.intino.consul.container.box.os.remote;

import com.jcraft.jsch.Session;
import io.intino.alexandria.logger.Logger;
import io.intino.consul.container.box.os.remote.linux.LinuxOSProcess;
import io.intino.consul.container.box.os.remote.linux.ProcPath;
import io.intino.consul.framework.Activity;
import io.intino.consul.framework.Activity.System.FileSystem;
import io.intino.consul.framework.Activity.System.OSProcess;
import io.intino.consul.framework.Activity.System.ProcessRunner;
import oshi.driver.linux.proc.Auxv;
import oshi.driver.linux.proc.UpTime;
import oshi.util.FileUtil;
import oshi.util.ParseUtil;

import java.io.*;
import java.nio.ByteBuffer;
import java.util.*;
import java.util.regex.Pattern;
import java.util.stream.Stream;

import static java.util.stream.Collectors.toSet;
import static oshi.util.ParseUtil.parseLongOrDefault;

public class RemoteOperatingSystem implements Activity.System.OperatingSystem {
	private final RemoteProcessRunner processRunner;
	private final RemoteFileSystem remoteFileSystem;
	private final long bootTime;
	private long userHz;
	private long pageSize;
	private static Set<String> systemProcesses;
	public RemoteOperatingSystem(Session session) throws IOException {
		this.remoteFileSystem = new RemoteFileSystem(session);
		this.processRunner = new RemoteProcessRunner(session);
		this.bootTime = booTime();
		if (systemProcesses == null || systemProcesses.isEmpty()) systemProcesses = systemProcesses();
		updateProperties();
	}

	@Override
	public Name name() {
		try {
			return processRunner().execute("uname").trim().equals("Linux") ? Name.Unix : Name.Mac;
		} catch (Exception e) {
			Logger.error(e);
			return Name.Unix;
		}
	}

	@Override
	public long bootTimeSeconds() {
		return this.bootTime;
	}

	@Override
	public FileSystem fileSystem() throws IOException {
		return remoteFileSystem;
	}

	@Override
	public ProcessRunner processRunner() {
		return processRunner;
	}

	@Override
	public List<? extends OSProcess> processes() throws IOException {
		return getPidFiles().map(f -> {
					try {
						return new LinuxOSProcess(f, this, systemProcesses);
					} catch (Exception e) {
						Logger.error(e);
						return null;
					}
				}).filter(Objects::nonNull)
				.toList();
	}

	private Stream<File> getPidFiles() throws IOException {
		File directory = new File(ProcPath.PROC);
		return fileSystem().listDirectory(directory.getAbsolutePath()).stream().filter(file -> DIGITS.matcher(file).matches()).map(n -> new File(directory, n));
	}

	public static final Pattern DIGITS = Pattern.compile("\\d+");

	@Override
	public int processCount() {
		return 0;
	}

	private long booTime() {
		long tempBT = bootTimeFromProc();
		if (tempBT == 0) tempBT = System.currentTimeMillis() / 1000L - (long) UpTime.getSystemUptimeSeconds();
		return tempBT;
	}

	public long bootTimeFromProc() {
		try {
			return fileSystem().readFile(ProcPath.STAT).lines()
					.filter(stat -> stat.startsWith("btime"))
					.findFirst()
					.map(t -> parseLongOrDefault(ParseUtil.whitespaces.split(t)[1], 0L))
					.orElse(0L);
		} catch (IOException e) {
			Logger.error(e);
			return 0;
		}
	}

	public long userHz() {
		return this.userHz;
	}

	public long pageSize() {
		return this.pageSize;
	}

	@Override
	public Activity.System.Measurements measurements() {
		return null;//TODO
	}

	private void updateProperties() {
		try {
			Map<Integer, Long> auxv = queryAuxv();
			long hz = auxv.getOrDefault(Auxv.AT_CLKTCK, 0L);
			userHz = hz > 0 ? hz : parseLongOrDefault(processRunner.execute("getconf", "CLK_TCK"), 100L);
			long pageSize = auxv.getOrDefault(Auxv.AT_PAGESZ, 0L);
			this.pageSize = pageSize > 0 ? pageSize : parseLongOrDefault(processRunner.execute("getconf", "PAGE_SIZE"), 4096L);
		} catch (Exception e) {
			Logger.error(e);
		}
	}

	private Map<Integer, Long> queryAuxv() {
		try {
			byte[] array = fileSystem().readFileBytes(ProcPath.AUXV);
			ByteBuffer buff = ByteBuffer.wrap(array);
			Map<Integer, Long> auxvMap = new HashMap<>();
			int key;
			do {
				key = FileUtil.readNativeLongFromBuffer(buff).intValue();
				if (key > 0) auxvMap.put(key, FileUtil.readNativeLongFromBuffer(buff).longValue());
			} while (key > 0);
			return auxvMap;
		} catch (IOException e) {
			Logger.error(e);
			return Map.of();
		}
	}

	private Set<String> systemProcesses() {
		InputStream stream = this.getClass().getClassLoader().getResourceAsStream(Name.Unix.name().toLowerCase() + ".system.processes.txt");
		if (stream == null) {
			Logger.error("Resource not found: " + Name.Unix.name().toLowerCase() + ".system.processes.txt");
			return Set.of();
		}
		return new BufferedReader(new InputStreamReader(stream)).lines().collect(toSet());
	}
}