package io.intino.consul.container.box.os.remote;

import com.jcraft.jsch.ChannelSftp;
import com.jcraft.jsch.JSchException;
import com.jcraft.jsch.Session;
import com.jcraft.jsch.SftpException;
import io.intino.alexandria.logger.Logger;
import io.intino.consul.framework.Activity;

import java.io.ByteArrayOutputStream;
import java.io.Closeable;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.file.CopyOption;
import java.nio.file.StandardOpenOption;
import java.util.Arrays;
import java.util.List;

import static java.nio.file.StandardOpenOption.APPEND;
import static java.nio.file.StandardOpenOption.CREATE;

public class RemoteFileSystem implements Activity.System.FileSystem, Closeable {
	private final Session session;
	private final ChannelSftp channel;

	public RemoteFileSystem(Session session) throws IOException {
		try {
			this.session = session;
			channel = openSftp();
		} catch (JSchException e) {
			throw new IOException(e.getMessage());
		}
	}

	@Override
	public boolean exists(String s) {
		return false;//TODO
	}

	@Override
	public String readFile(String path) {
		try {
			ByteArrayOutputStream stream = new ByteArrayOutputStream();
			channel.get(path, stream);
			return stream.toString();
		} catch (SftpException e) {
			Logger.error(e);
			return null;
		}
	}

	@Override
	public byte[] readFileBytes(String s) {
		return new byte[0];
	}

	@Override
	public void writeString(String path, String content) {
		writeString(path, content, CREATE);
	}

	@Override
	public void writeString(String path, String content, StandardOpenOption... options) {
		try {
			OutputStream outputStream = channel.put(path, Arrays.asList(options).contains(APPEND) ? ChannelSftp.APPEND : ChannelSftp.OVERWRITE);
			outputStream.write(content.getBytes());
			outputStream.close();
		} catch (SftpException | IOException e) {
			Logger.error(e.getMessage() + ". " + path);
		}
	}

	@Override
	public void changeOwner(String path, String user, String group) {
		try {
			new RemoteProcessRunner(session).execute("chown", "-R", user + ":" + group, path);
		} catch (Exception e) {
			Logger.error(e.getMessage() + ". " + path);
		}
	}

	@Override
	public void copy(String source, String target, CopyOption... options) {
		try {
			channel.put(source, target);
		} catch (SftpException e) {
			Logger.error(e.getMessage() + ". source: " + source + "; target: " + target);
		}
	}

	@Override
	public void mkdirs(String path) {
		try {
			new RemoteProcessRunner(session).execute("mkdir", "-p", path);
		} catch (Exception e) {
			Logger.error(e.getMessage() + ". " + path);
		}
	}

	@Override
	public void delete(String path) {
		try {
			new RemoteProcessRunner(session).execute("rm -f", path);
		} catch (Exception e) {
			Logger.error(e);
		}
	}

	@Override
	public void deleteDirectory(String path) {
		try {
			new RemoteProcessRunner(session).execute("rm", "-Rf", path);
		} catch (Exception e) {
			Logger.error(e.getMessage() + ". " + path);
		}
	}

	@Override
	public List<String> listDirectory(String s) {
		return null;
	}

	private ChannelSftp openSftp() throws JSchException {
		ChannelSftp channel = (ChannelSftp) session.openChannel("sftp");
		channel.connect();
		return channel;
	}

	@Override
	public void close() {
		if (channel != null) channel.disconnect();
	}
}
