package io.intino.consul.container.box.os.local;

import io.intino.alexandria.logger.Logger;
import io.intino.consul.framework.Activity;
import io.intino.consul.framework.Activity.System.ProcessRunner;
import org.apache.log4j.Level;
import org.apache.log4j.LogManager;
import oshi.SystemInfo;
import oshi.hardware.CentralProcessor;
import oshi.hardware.GlobalMemory;
import oshi.hardware.NetworkIF;
import oshi.software.os.OSProcess;
import oshi.software.os.OSService;

import java.io.*;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import static io.intino.consul.framework.Activity.System.OperatingSystem.Name.Unix;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;

public class LocalOperatingSystem implements Activity.System.OperatingSystem {
	private final Name name;
	private final SystemInfo si;
	private static Set<String> systemProcesses;

	public LocalOperatingSystem() {
		String os = java.lang.System.getProperty("os.name").toLowerCase();
		name = os.contains("win") ? Name.Windows : (os.contains("mac") ? Name.Mac : Unix);
		if (systemProcesses == null || systemProcesses.isEmpty()) systemProcesses = systemProcesses();
		si = new SystemInfo();
		LogManager.getCurrentLoggers().asIterator().forEachRemaining(c -> ((org.apache.log4j.Logger) c).setLevel(Level.ERROR));
	}

	private Set<String> systemProcesses() {
		InputStream stream = this.getClass().getClassLoader().getResourceAsStream(name.name().toLowerCase() + ".system.processes.txt");
		if (stream == null) {
			Logger.error("Resource not found: " + name.name().toLowerCase() + ".system.processes.txt");
			return Set.of();
		}
		return new BufferedReader(new InputStreamReader(stream)).lines().collect(toSet());
	}

	@Override
	public Name name() {
		return name;
	}

	@Override
	public long bootTimeSeconds() {
		return si.getOperatingSystem().getSystemBootTime();
	}

	@Override
	public long userHz() {
		return si.getHardware().getProcessor().getMaxFreq();
	}

	@Override
	public long pageSize() {
		return si.getHardware().getMemory().getPageSize();
	}

	@Override
	public Activity.System.Measurements measurements() {
		return new Activity.System.Measurements() {
			@Override
			public long usageRAM() {
				GlobalMemory memory = si.getHardware().getMemory();
				return inMb(memory.getTotal() - memory.getAvailable());
			}

			@Override
			public long usageHDD() {
				File file = new File(".");
				return inMb(file.getTotalSpace() - file.getFreeSpace());
			}

			@Override
			public double usageCPU() {
				CentralProcessor cpu = si.getHardware().getProcessor();
				return asPercent(cpu.getSystemLoadAverage(3)[2]);
			}

			@Override
			public int usageFiles() {
				try {
					return (int) si.getOperatingSystem().getProcesses().stream().mapToLong(OSProcess::getOpenFiles).sum();
				} catch (Throwable e) {
					return 0;
				}
			}

			@Override
			public long usageThreads() {
				return si.getOperatingSystem().getThreadCount();
			}

			@Override
			public long dataReceived() {
				NetworkIF networkIF = si.getHardware().getNetworkIFs().stream().filter(i -> i.getIfOperStatus().equals(NetworkIF.IfOperStatus.UP) && i.getIPv4addr().length > 0).findFirst().orElse(null);
				if (networkIF == null) return 0;
				networkIF.updateAttributes();
				return networkIF.getBytesRecv() / (1024 * 1024);
			}

			@Override
			public long dataSent() {
				NetworkIF networkIF = si.getHardware().getNetworkIFs().stream().filter(i -> i.getIPv4addr().length > 0).findFirst().orElse(null);
				if (networkIF == null) return 0;
				networkIF.updateAttributes();
				return networkIF.getBytesSent() / (1024 * 1024);
			}

			@Override
			public double temperatureKernel() {
				return readTemperature();
			}

			@Override
			public double temperatureExternal() {
				return 0;
			}
		};
	}

	private double readTemperature() {
		try {
			if (name == Unix && new File("/opt/vc/bin/vcgencmd").exists()) return readTemperatureInUnixOS();
			return si.getHardware().getSensors().getCpuTemperature();
		} catch (Exception ignored) {
		}
		return 0;
	}

	private static double readTemperatureInUnixOS() throws IOException, InterruptedException {
		Process p = Runtime.getRuntime().exec("/opt/vc/bin/vcgencmd measure_temp");
		try (BufferedReader br = new BufferedReader(new InputStreamReader(p.getInputStream()))) {
			String s;
			while ((s = br.readLine()) != null)
				return Float.parseFloat(s.replace("temp=", "").replace("'C", ""));
			p.waitFor();
			return 0.;
		}
	}

	public static long inMb(long value) {
		return value / (1024 * 1024);
	}

	private static double asPercent(double number) {
		return Math.floor(number * 100) / 100;
	}

	@Override
	public Activity.System.FileSystem fileSystem() {
		return new LocalFileSystem(this);
	}

	@Override
	public ProcessRunner processRunner() {
		return new LocalProcessRunner();
	}

	@Override
	public List<Activity.System.OSProcess> processes() {
		Map<Integer, String> services = systemServices();
		if (name == Unix) searchJavaServices(services);
		return si.getOperatingSystem().getProcesses().stream()
				.map(p -> map(p, services.get(p.getProcessID())))
				.toList();
	}

	private void searchJavaServices(Map<Integer, String> services) {
		ProcessRunner runner = processRunner();
		try {
			Map<Integer, String> serviceNames = run(runner, "systemctl", "--type=service", "--state=running").lines()
					.filter(l -> l.contains(".service"))
					.map(l -> l.substring(0, l.indexOf(".service")).trim())
					.collect(Collectors.toMap(l -> parse(run(runner, "systemctl", "show", "--value", "--property", "MainPID", l)), l -> l));
			services.putAll(serviceNames);
		} catch (Exception e) {
			Logger.error(e);
		}
	}

	private Integer parse(String pid) {
		try {
			return Integer.parseInt(pid.trim());
		} catch (NumberFormatException e) {
			Logger.error(e);
			return 0;
		}
	}

	private static String run(ProcessRunner runner, String... command) {
		try {
			return runner.execute(command);
		} catch (Exception e) {
			Logger.error(e);
			return "";
		}
	}

	private Map<Integer, String> systemServices() {
		return si.getOperatingSystem().getServices().stream()
				.collect(toMap(OSService::getProcessID, OSService::getName, (v1, v2) -> v2));
	}

	@Override
	public int processCount() {
		return si.getOperatingSystem().getProcessCount();
	}

	private Activity.System.OSProcess map(OSProcess p, String serviceName) {
		return new Activity.System.OSProcess() {
			@Override
			public String name() {
				return p.getName();
			}

			@Override
			public String path() {
				return p.getPath();
			}

			@Override
			public String commandLine() {
				return p.getCommandLine();
			}

			@Override
			public List<String> arguments() {
				return p.getArguments();
			}

			@Override
			public String user() {
				return p.getUser();
			}

			@Override
			public String group() {
				return p.getGroup();
			}

			@Override
			public State state() {
				return Activity.System.OSProcess.State.valueOf(p.getState().name());
			}

			@Override
			public boolean isOSProcess() {
				String name = p.getName();
				if (systemProcesses.contains(name) || systemProcesses.contains(name.split("/")[0])) return true;
				if (LocalOperatingSystem.this.name == Name.Windows) return p.getCommandLine().startsWith("C:\\Windows");
				if (LocalOperatingSystem.this.name == Name.Mac) return p.getCommandLine().startsWith("/System/");
				return false;
			}

			@Override
			public boolean isSystemService() {
				return serviceName != null;
			}

			@Override
			public String systemServiceName() {
				return serviceName;
			}


			@Override
			public int processID() {
				return p.getProcessID();
			}

			@Override
			public int parentProcessID() {
				return p.getParentProcessID();
			}

			@Override
			public int threadCount() {
				return p.getThreadCount();
			}

			@Override
			public long virtualSize() {
				return p.getVirtualSize();
			}

			@Override
			public long residentSetSize() {
				return p.getResidentSetSize();
			}

			@Override
			public long kernelTime() {
				return p.getKernelTime();
			}

			@Override
			public long userTime() {
				return p.getUserTime();
			}

			@Override
			public long upTime() {
				return p.getUpTime();
			}

			@Override
			public long startTime() {
				return p.getStartTime();
			}

			@Override
			public long bytesRead() {
				return p.getBytesRead();
			}

			@Override
			public long bytesWritten() {
				return p.getBytesWritten();
			}

			@Override
			public long openFiles() {
				return p.getOpenFiles();
			}

			@Override
			public long openFilesLimit() {
				return p.getHardOpenFileLimit();
			}

			@Override
			public double processCpuLoadCumulative() {
				return p.getProcessCpuLoadCumulative();
			}

			@Override
			public int getBitness() {
				return p.getBitness();
			}
		};
	}
}
