package io.intino.consul.container.box.os.local;

import io.intino.alexandria.logger.Logger;
import io.intino.consul.framework.Activity;
import org.apache.commons.io.FileUtils;

import java.io.File;
import java.io.IOException;
import java.nio.file.*;
import java.nio.file.attribute.GroupPrincipal;
import java.nio.file.attribute.PosixFileAttributeView;
import java.nio.file.attribute.UserPrincipalLookupService;
import java.util.Collections;
import java.util.List;
import java.util.stream.Stream;

import static io.intino.consul.framework.Activity.System.OperatingSystem.Name.Unix;
import static java.nio.file.StandardOpenOption.CREATE;

public class LocalFileSystem implements Activity.System.FileSystem {
	private final LocalOperatingSystem localOperatingSystem;

	public LocalFileSystem(LocalOperatingSystem localOperatingSystem) {
		this.localOperatingSystem = localOperatingSystem;
	}

	@Override
	public boolean exists(String path) {
		return new File(path).exists();
	}

	@Override
	public String readFile(String path) {
		try {
			return Files.readString(Path.of(path));
		} catch (IOException e) {
			Logger.error(e);
			return null;
		}
	}

	@Override
	public byte[] readFileBytes(String s) {
		return new byte[0];
	}

	@Override
	public void writeString(String path, String content) {
		writeString(path, content, CREATE);
	}

	@Override
	public void writeString(String path, String content, StandardOpenOption... options) {
		try {
			Files.writeString(Path.of(path), content, options);
		} catch (IOException e) {
			Logger.error(e);
		}
	}

	public void changeOwner(String path, String user, String group) {
		if (localOperatingSystem.name() != Unix) return;
		Path filePath = Path.of(path);
		UserPrincipalLookupService lookupService = filePath.getFileSystem().getUserPrincipalLookupService();
		try (Stream<Path> paths = Files.walk(filePath)) {
			paths.forEach(p -> {
				try {
					Files.setOwner(filePath, lookupService.lookupPrincipalByName(user));
					GroupPrincipal groupPrincipal = lookupService.lookupPrincipalByGroupName(group);
					if (groupPrincipal != null) {
						PosixFileAttributeView fileAttributeView = Files.getFileAttributeView(filePath, PosixFileAttributeView.class, LinkOption.NOFOLLOW_LINKS);
						fileAttributeView.setGroup(groupPrincipal);
					}
				} catch (IOException e) {
					Logger.error(e);
				}
			});
		} catch (IOException e) {
			Logger.error(e);
		}
	}

	@Override
	public void copy(String source, String target, CopyOption... options) {
		try {
			new File(target).getParentFile().mkdirs();
			Files.copy(Path.of(source), Path.of(target), options);
		} catch (IOException e) {
			Logger.error(e);
		}
	}

	@Override
	public void mkdirs(String path) {
		new File(path).mkdirs();
	}

	@Override
	public void delete(String path) {
		try {
			Files.delete(Path.of(path));
		} catch (IOException e) {
			Logger.error(e);
		}

	}

	@Override
	public void deleteDirectory(String path) {
		try {
			FileUtils.deleteDirectory(new File(path));
		} catch (IOException e) {
			Logger.error(e);
		}
	}

	@Override
	public List<String> listDirectory(String path) {
		try {
			return Files.list(Path.of(path)).map(p -> p.toFile().getName()).toList();
		} catch (IOException e) {
			Logger.error(e);
			return Collections.emptyList();
		}
	}
}
