package io.intino.consul.container.box.activity;

import io.intino.alexandria.logger.Logger;
import io.intino.consul.container.box.ContainerConfiguration;
import io.intino.consul.container.box.ContainerStore;
import io.intino.consul.container.box.activity.ActivityLoader.JarClassLoader;
import io.intino.consul.container.box.os.local.LocalSystem;
import io.intino.consul.container.box.os.remote.RemoteSystem;
import io.intino.consul.framework.Activity;
import io.intino.consul.framework.Activity.Context;

import java.io.File;
import java.io.IOException;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static io.intino.consul.container.box.ContainerStore.asBooleanMap;

public class ActivityManager {
	public static final String ROOT_ACTIVITIES = "root#activities";
	private final HashMap<String, Activity> enabledActivities;
	private final ContainerConfiguration configuration;
	private final Context context;
	private final ContainerStore store;
	private final ActivityLoader loader;

	public ActivityManager(ContainerConfiguration configuration, Context context, ContainerStore store) {
		this.configuration = configuration;
		this.context = context;
		this.store = store;
		this.loader = new ActivityLoader();
		this.enabledActivities = new HashMap<>();
	}

	public List<String> enabledActivities() {
		return new ArrayList<>(enabledActivities.keySet());
	}

	public void loadActivitiesFromStore() {
		Map<String, Boolean> activities = activities();
		if (activities == null) return;
		activities.keySet().stream()
				.filter(activities::get)
				.forEach(this::enableActivity);
	}

	public void stopActivities() {
		enabledActivities.values().forEach(ActivityManager::stop);
	}

	public boolean enableActivity(String activityId) {
		Activity activity = enabledActivities.get(activityId);
		if (activity != null) return true;
		activity = loader.load(store.activity(activityId));
		if (activity == null) return false;
		boolean enabled = configure(activity, store.activityParams(activityId));
		if (!enabled) return false;
		enabledActivities.put(activityId, activity);
		update(activityId, true);
		Logger.info("Enabled activity " + activityId);
		new Thread(activity::onStart).start();
		return true;
	}

	public boolean disableActivity(String activityId) {
		Activity activity = enabledActivities.get(activityId);
		if (activity == null) return false;
		stop(activity);
		enabledActivities.remove(activityId);
		update(activityId, false);
		Logger.info("Disabled activity " + activityId);
		return true;
	}

	public boolean updateConfiguration(String activityId, Map<String, String> configuration) {
		if (!activities().containsKey(activityId)) return false;
		store.saveActivityParams(activityId, configuration);
		Logger.info("Updated configuration of " + activityId);
		return true;
	}

	public synchronized void installActivity(String activityId, byte[] payload) {
		Map<String, Boolean> activities = activities();
		if (!activities.containsKey(activityId)) activities.put(activityId, false);
		store.put(ROOT_ACTIVITIES, activities);
		store.saveActivity(activityId, payload);
		Logger.info("Installed activity " + activityId);
	}

	public synchronized boolean uninstallActivity(String activityId) {
		Map<String, Boolean> activities = activities();
		if (activities == null) return false;
		activities.remove(activityId);
		store.put(ROOT_ACTIVITIES, activities);
		remove(activityId, this.enabledActivities.remove(activityId));
		store.save();
		Logger.info("Uninstalled activity " + activityId);
		return true;
	}

	public synchronized void saveConfigurations() {
		enabledActivities.forEach((k, v) -> store.saveActivityParams(k, v.currentConfiguration()));
	}

	private void update(String activityId, boolean value) {
		Map<String, Boolean> activities = activities();
		activities.put(activityId, value);
		store.put(ROOT_ACTIVITIES, activities);
	}

	private void remove(String activityId, Activity activity) {
		if (activity != null) {
			stop(activity);
			clearStore(activity.id());
			closeClassLoader(activity);
			stopActivityThreads(activityId);
			java.lang.System.gc();
		}
		store.removeActivity(activityId);
	}

	private void stopActivityThreads(String artifact) {
		Thread.getAllStackTraces().keySet().stream()
				.filter(t -> t.getContextClassLoader() instanceof URLClassLoader &&
						new File(((URLClassLoader) t.getContextClassLoader()).getURLs()[0].getFile()).equals(store.activity(artifact)))
				.forEach(Thread::interrupt);
	}

	private boolean configure(Activity a, Map<String, String> args) {
		try {
			return a.configure(new Context(context.terminal(), args.containsKey("remote-server") ? remoteSystem(args) : new LocalSystem(configuration.ispProviderToken()), context.hostName(), context.observer(), context.ss(), mergeArgs(args, context.initialConfiguration())), new ActivityStore(a.id(), store));
		} catch (Throwable e) {
			Logger.error(e);
			return false;
		}
	}

	private Activity.System remoteSystem(Map<String, String> args) throws Exception {
		return new RemoteSystem(args.get("private-key"), args.get("remote-server"), args.get("remote-port"), args.get("remote-user"));

	}

	private Map<String, String> mergeArgs(Map<String, String> params, Map<String, String> args) {
		HashMap<String, String> merged = new HashMap<>(params);
		merged.putAll(args);
		return merged;
	}

	private static void stop(Activity activity) {
		try {
			activity.onStop();
		} catch (Throwable e) {
			Logger.error(e);
		}
	}

	public Map<String, Boolean> activities() {
		return store.getOrElse(ROOT_ACTIVITIES, asBooleanMap, new HashMap<>());
	}

	private void clearStore(String id) {
		store.removeAll(k -> k.startsWith(id + Activity.Store.Separator));
	}

	private static void closeClassLoader(Activity activity) {
		ClassLoader classLoader = activity.getClass().getClassLoader();
		if (classLoader instanceof JarClassLoader) {
			classLoader.clearAssertionStatus();
			try {
				((JarClassLoader) classLoader).close();
			} catch (IOException e) {
				Logger.error(e);
			}
		}
	}
}