package io.intino.consul.container.box;

import com.google.gson.reflect.TypeToken;
import io.intino.alexandria.Json;
import io.intino.alexandria.logger.Logger;
import io.intino.consul.framework.Activity.Store;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.lang.reflect.Type;
import java.nio.file.Files;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Predicate;
import java.util.stream.Stream;

import static java.io.File.separator;
import static java.util.stream.Collectors.toMap;

public class ContainerStore implements Store {
	public static final Type asList = new TypeToken<List<String>>() {
	}.getType();
	public static final Type asMap = new TypeToken<Map<String, String>>() {
	}.getType();
	public static final Type asBooleanMap = new TypeToken<Map<String, Boolean>>() {
	}.getType();
	private static final String SEP = "\t";
	private final File directory;
	private final File activitiesDirectory;
	private Map<String, String> entries;

	public ContainerStore(File directory) {
		this.directory = directory;
		Logger.info("Container store located in: " + directory.getAbsolutePath());
		directory.mkdirs();
		activitiesDirectory = new File(directory, "activities");
		activitiesDirectory.mkdirs();
		load();
	}

	public File saveActivity(String artifact, byte[] content) {
		File file = activityFile(artifact);
		file.getParentFile().mkdirs();
		try {
			Files.write(file.toPath(), content);
			return file;
		} catch (IOException e) {
			Logger.error(e);
			return file;
		}
	}

	public File saveActivityParams(String artifact, Map<String, String> params) {
		File paramsFile = parametersFile(artifact);
		try {
			Files.writeString(paramsFile.toPath(), Json.toJson(params));
			return paramsFile;
		} catch (IOException e) {
			Logger.error(e);
			return paramsFile;
		}
	}


	public void removeActivity(String artifact) {
		File file = activityFile(artifact);
		if (!file.delete()) file.deleteOnExit();
		parametersFile(artifact).delete();
	}

	public File activity(String artifact) {
		return activityFile(artifact);
	}

	public Map<String, String> activityParams(String artifact) {
		try {
			File file = parametersFile(artifact);
			if (!file.exists()) return Map.of();
			return Json.fromJson(Files.readString(file.toPath()), asMap);
		} catch (IOException e) {
			Logger.error(e);
			return Map.of();
		}
	}

	private void load() {
		if (!file().exists()) entries = new HashMap<>();
		else try (Stream<String> lines = Files.lines(file().toPath())) {
			entries = lines.map(l -> l.split(SEP))
					.collect(toMap(f -> f[0], f -> f[1]));
		} catch (IOException e) {
			Logger.error(e);
		}
	}

	public synchronized void save() {
		try {
			file().delete();
			if (!entries.isEmpty()) {
				BufferedWriter writer = new BufferedWriter(new FileWriter(file(), true));
				entries.entrySet().stream()
						.map(e -> e.getKey() + SEP + e.getValue() + "\n")
						.forEach(l -> {
							try {
								writer.write(l);
							} catch (IOException e) {
								Logger.error(e);
							}
						});
				writer.close();
			}
		} catch (IOException e) {
			Logger.error(e);
		}
	}

	@Override
	public void put(String key, Object object) {
		entries.put(key, Json.toJson(object));
	}

	@Override
	public <T> T get(String key, Class<T> type) {
		String value = entries.get(key);
		return value == null ? null : Json.fromJson(value, type);
	}

	public <T> T getOrElse(String key, Class<T> type, T orElse) {
		String value = entries.get(key);
		return value == null ? orElse : Json.fromJson(value, type);
	}

	@Override
	public <T> T get(String key, Type type) {
		String value = entries.get(key);
		return value == null ? null : Json.fromJson(value, type);
	}

	public <T> T getOrElse(String key, Type type, T orElse) {
		String value = entries.get(key);
		return value == null ? orElse : Json.fromJson(value, type);
	}

	private File activityFile(String artifact) {
		String[] coors = artifact.split(":");
		return new File(activitiesDirectory, coors[0] + separator + coors[1] + separator + coors[2] + separator + coors[1] + "-" + coors[2] + ".jar");
	}

	private File parametersFile(String artifact) {
		String[] coors = artifact.split(":");
		return new File(activitiesDirectory, coors[0] + separator + coors[1] + separator + coors[2] + separator + "parameters.json");
	}

	private File file() {
		return new File(directory, "store.json");
	}

	public Stream<String> keys() {
		return entries.keySet().stream();
	}

	public void remove(String key) {
		entries.remove(key);
	}

	public void removeAll(Predicate<String> predicate) {
		entries.keySet().removeIf(predicate);
	}

	public void clear() {
		entries.clear();
	}
}
