/*
 * Decompiled with CFR 0.152.
 */
package io.intino.alexandria.terminal;

import io.intino.alexandria.event.Event;
import io.intino.alexandria.jms.BusConnector;
import io.intino.alexandria.jms.ConnectionListener;
import io.intino.alexandria.jms.JmsConsumer;
import io.intino.alexandria.jms.JmsProducer;
import io.intino.alexandria.jms.MessageReader;
import io.intino.alexandria.jms.MessageWriter;
import io.intino.alexandria.jms.QueueConsumer;
import io.intino.alexandria.jms.QueueProducer;
import io.intino.alexandria.jms.TopicConsumer;
import io.intino.alexandria.jms.TopicProducer;
import io.intino.alexandria.logger.Logger;
import io.intino.alexandria.message.Message;
import io.intino.alexandria.terminal.Broker;
import io.intino.alexandria.terminal.Connector;
import io.intino.alexandria.terminal.EventOutBox;
import io.intino.alexandria.terminal.MessageOutBox;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import javax.jms.Connection;
import javax.jms.JMSException;
import javax.jms.MessageConsumer;
import javax.jms.Session;
import javax.jms.TemporaryQueue;
import javax.jms.TextMessage;
import org.apache.activemq.ActiveMQConnection;
import org.apache.activemq.ActiveMQSession;
import org.apache.activemq.command.ActiveMQDestination;

public class JmsConnector
implements Connector {
    private final Map<String, JmsProducer> producers;
    private final Map<String, JmsConsumer> consumers;
    private final Map<String, List<Consumer<Event>>> eventConsumers;
    private final Map<String, List<Connector.MessageConsumer>> messageConsumers;
    private final Map<Consumer<Event>, Integer> jmsEventConsumers;
    private final Map<Connector.MessageConsumer, Integer> jmsMessageConsumers;
    private final String brokerUrl;
    private final String user;
    private final String password;
    private final String clientId;
    private final boolean transactedSession;
    private final AtomicBoolean connected = new AtomicBoolean(false);
    private final AtomicBoolean started = new AtomicBoolean(false);
    private EventOutBox eventOutBox;
    private MessageOutBox messageOutBox;
    private Connection connection;
    private Session session;
    private ScheduledExecutorService scheduler;

    public JmsConnector(String brokerUrl, String user, String password, String clientId, File messageCacheDirectory) {
        this(brokerUrl, user, password, clientId, false, messageCacheDirectory);
    }

    public JmsConnector(String brokerUrl, String user, String password, String clientId, boolean transactedSession, File outBoxDirectory) {
        this.brokerUrl = brokerUrl;
        this.user = user;
        this.password = password;
        this.clientId = clientId;
        this.transactedSession = transactedSession;
        this.producers = new HashMap<String, JmsProducer>();
        this.consumers = new HashMap<String, JmsConsumer>();
        this.jmsEventConsumers = new HashMap<Consumer<Event>, Integer>();
        this.jmsMessageConsumers = new HashMap<Connector.MessageConsumer, Integer>();
        this.eventConsumers = new HashMap<String, List<Consumer<Event>>>();
        this.messageConsumers = new HashMap<String, List<Connector.MessageConsumer>>();
        if (outBoxDirectory != null) {
            this.eventOutBox = new EventOutBox(new File(outBoxDirectory, "events"));
            this.messageOutBox = new MessageOutBox(new File(outBoxDirectory, "requests"));
        }
    }

    public void start() {
        if (this.brokerUrl == null || this.brokerUrl.isEmpty()) {
            Logger.warn("Invalid broker URL. Connection aborted");
            return;
        }
        try {
            this.connect();
        }
        catch (JMSException e) {
            Logger.error(e);
        }
        this.started.set(true);
        if (this.scheduler == null) {
            this.scheduler = Executors.newScheduledThreadPool(1);
            this.scheduler.scheduleAtFixedRate(this::checkConnection, 15L, 10L, TimeUnit.MINUTES);
        }
    }

    private void connect() throws JMSException {
        if (!Broker.isRunning(this.brokerUrl)) {
            Logger.warn("Broker Unreachable. Connection aborted");
            return;
        }
        this.initConnection();
        if (this.connection != null && ((ActiveMQConnection)this.connection).isStarted()) {
            this.clearProducers();
            this.session = this.createSession(this.transactedSession);
            if (this.session != null && ((ActiveMQSession)this.session).isRunning()) {
                this.connected.set(true);
                this.recoverEventsAndMessages();
            }
        }
    }

    @Override
    public synchronized void sendEvent(String path, Event event) {
        new ArrayList<Consumer>(this.eventConsumers.getOrDefault(path, Collections.emptyList())).forEach(eventConsumer -> eventConsumer.accept(event));
        if (!this.doSendEvent(path, event) && this.eventOutBox != null) {
            this.eventOutBox.push(path, event);
        }
    }

    @Override
    public void attachListener(String path, Consumer<Event> onEventReceived) {
        this.registerEventConsumer(path, onEventReceived);
        JmsConsumer consumer = this.consumers.get(path);
        if (consumer == null) {
            return;
        }
        Consumer<javax.jms.Message> eventConsumer = e -> onEventReceived.accept(new Event(MessageDeserializer.deserialize(e)));
        this.jmsEventConsumers.put(onEventReceived, eventConsumer.hashCode());
        consumer.listen(eventConsumer);
    }

    @Override
    public void sendMessage(String path, String message) {
        this.recoverEventsAndMessages();
        if (!this.doSendMessage(path, message) && this.messageOutBox != null) {
            this.messageOutBox.push(path, message);
        }
    }

    @Override
    public void attachListener(String path, String subscriberId, Consumer<Event> onEventReceived) {
        this.registerEventConsumer(path, onEventReceived);
        TopicConsumer consumer = (TopicConsumer)this.consumers.get(path);
        if (consumer == null) {
            return;
        }
        Consumer<javax.jms.Message> eventConsumer = m -> onEventReceived.accept(new Event(MessageDeserializer.deserialize(m)));
        this.jmsEventConsumers.put(onEventReceived, eventConsumer.hashCode());
        consumer.listen(eventConsumer, subscriberId);
    }

    @Override
    public void detachListeners(Consumer<Event> consumer) {
        Integer code = this.jmsEventConsumers.get(consumer);
        if (code == null) {
            return;
        }
        for (JmsConsumer jc : this.consumers.values()) {
            List<Consumer> toRemove = jc.listeners().stream().filter(l -> l.hashCode() == code.intValue()).collect(Collectors.toList());
            toRemove.forEach(jc::removeListener);
        }
        this.eventConsumers.values().forEach(list -> list.remove(consumer));
    }

    @Override
    public void attachListener(String path, Connector.MessageConsumer onMessageReceived) {
        this.registerMessageConsumer(path, onMessageReceived);
        JmsConsumer consumer = this.consumers.get(path);
        if (consumer == null) {
            return;
        }
        Consumer<javax.jms.Message> messageConsumer = m -> onMessageReceived.accept(MessageReader.textFrom(m), this.callback((javax.jms.Message)m));
        this.jmsMessageConsumers.put(onMessageReceived, messageConsumer.hashCode());
        consumer.listen(messageConsumer);
    }

    @Override
    public void detachListeners(Connector.MessageConsumer consumer) {
        Integer code = this.jmsMessageConsumers.get(consumer);
        if (code == null) {
            return;
        }
        for (JmsConsumer jc : this.consumers.values()) {
            List<Consumer> toRemove = jc.listeners().stream().filter(l -> l.hashCode() == code.intValue()).collect(Collectors.toList());
            toRemove.forEach(jc::removeListener);
        }
        this.messageConsumers.values().forEach(list -> list.remove(consumer));
    }

    @Override
    public void detachListeners(String path) {
        if (this.consumers.containsKey(path)) {
            this.consumers.get(path).close();
            this.consumers.remove(path);
            this.eventConsumers.get(path).clear();
            this.messageConsumers.get(path).clear();
        }
    }

    @Override
    public void requestResponse(String path, String message, Consumer<String> onResponse) {
        if (this.session == null) {
            Logger.error("Session is null");
            return;
        }
        try {
            QueueProducer producer = new QueueProducer(this.session, path);
            TemporaryQueue temporaryQueue = this.session.createTemporaryQueue();
            MessageConsumer consumer = this.session.createConsumer(temporaryQueue);
            consumer.setMessageListener(m -> this.acceptMessage(onResponse, consumer, (TextMessage)m));
            TextMessage txtMessage = this.session.createTextMessage();
            txtMessage.setText(message);
            txtMessage.setJMSReplyTo(temporaryQueue);
            txtMessage.setJMSCorrelationID(JmsConnector.createRandomString());
            this.sendMessage(producer, txtMessage);
            producer.close();
        }
        catch (JMSException e) {
            Logger.error(e);
        }
    }

    @Override
    public void requestResponse(String path, String message, String responsePath) {
        if (!this.doSendMessage(path, message, responsePath)) {
            this.messageOutBox.push(path, message);
        }
    }

    public Connection connection() {
        return this.connection;
    }

    public Session session() {
        return this.session;
    }

    public void stop() {
        try {
            this.consumers.values().forEach(JmsConsumer::close);
            this.consumers.clear();
            this.producers.values().forEach(JmsProducer::close);
            this.producers.clear();
            if (this.session != null) {
                this.session.close();
            }
            if (this.connection != null) {
                this.connection.close();
            }
            this.session = null;
            this.connection = null;
        }
        catch (Exception e) {
            Logger.error(e);
        }
    }

    private Session createSession(boolean transactedSession) throws JMSException {
        return this.connection.createSession(transactedSession, transactedSession ? 0 : 1);
    }

    private void registerEventConsumer(String path, Consumer<Event> onEventReceived) {
        this.eventConsumers.putIfAbsent(path, new CopyOnWriteArrayList());
        this.eventConsumers.get(path).add(onEventReceived);
        if (this.session != null && !this.consumers.containsKey(path)) {
            this.consumers.put(path, this.topicConsumer(path));
        }
    }

    private void registerMessageConsumer(String path, Connector.MessageConsumer onMessageReceived) {
        this.messageConsumers.putIfAbsent(path, new CopyOnWriteArrayList());
        this.messageConsumers.get(path).add(onMessageReceived);
        if (this.session != null && !this.consumers.containsKey(path)) {
            this.consumers.put(path, this.queueConsumer(path));
        }
    }

    private boolean doSendEvent(String path, Event event) {
        if (this.cannotSendMessage()) {
            return false;
        }
        try {
            this.topicProducer(path);
            JmsProducer producer = this.producers.get(path);
            return this.sendMessage(producer, JmsConnector.serialize(event));
        }
        catch (IOException | JMSException e) {
            Logger.error(e);
            return false;
        }
    }

    private boolean doSendMessage(String path, String message) {
        if (this.cannotSendMessage()) {
            return false;
        }
        try {
            this.queueProducer(path);
            JmsProducer producer = this.producers.get(path);
            return this.sendMessage(producer, JmsConnector.serialize(message));
        }
        catch (IOException | JMSException e) {
            Logger.error(e);
            return false;
        }
    }

    private boolean doSendMessage(String path, String message, String replyTo) {
        if (this.cannotSendMessage()) {
            return false;
        }
        try {
            this.queueProducer(path);
            JmsProducer producer = this.producers.get(path);
            return this.sendMessage(producer, this.serialize(message, replyTo));
        }
        catch (IOException | JMSException e) {
            Logger.error(e);
            return false;
        }
    }

    private void topicProducer(String path) throws JMSException {
        if (!this.producers.containsKey(path)) {
            this.producers.put(path, new TopicProducer(this.session, path));
        }
    }

    private void queueProducer(String path) throws JMSException {
        if (!this.producers.containsKey(path)) {
            this.producers.put(path, new QueueProducer(this.session, path));
        }
    }

    private boolean cannotSendMessage() {
        return this.session == null || !this.connected.get();
    }

    private boolean sendMessage(JmsProducer producer, javax.jms.Message message) {
        boolean[] result = new boolean[]{false};
        try {
            Thread thread = new Thread(() -> {
                result[0] = producer.produce(message);
            });
            thread.start();
            thread.join(1000L);
            thread.interrupt();
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
        return result[0];
    }

    private void acceptMessage(Consumer<String> onResponse, MessageConsumer consumer, TextMessage m) {
        try {
            onResponse.accept(m.getText());
            consumer.close();
        }
        catch (JMSException e) {
            Logger.error(e);
        }
    }

    private ConnectionListener connectionListener() {
        return new ConnectionListener(){

            @Override
            public void transportInterupted() {
                Logger.warn("Connection with Data Hub interrupted!");
                JmsConnector.this.connected.set(false);
            }

            @Override
            public void transportResumed() {
                Logger.info("Connection with Data Hub established!");
                JmsConnector.this.connected.set(true);
                JmsConnector.this.recoverConsumers();
            }
        };
    }

    private void clearProducers() {
        this.producers.values().forEach(JmsProducer::close);
        this.producers.clear();
    }

    private void clearConsumers() {
        this.consumers.values().forEach(JmsConsumer::close);
        this.consumers.clear();
    }

    private TopicConsumer topicConsumer(String path) {
        try {
            return new TopicConsumer(this.session, path);
        }
        catch (JMSException e) {
            Logger.error(e);
            return null;
        }
    }

    private QueueConsumer queueConsumer(String path) {
        try {
            return new QueueConsumer(this.session, path);
        }
        catch (JMSException e) {
            Logger.error(e);
            return null;
        }
    }

    private void recoverConsumers() {
        if (!this.started.get()) {
            return;
        }
        if (!this.eventConsumers.isEmpty() && this.consumers.isEmpty()) {
            for (String path : this.eventConsumers.keySet()) {
                this.consumers.put(path, this.topicConsumer(path));
            }
        }
        if (!this.messageConsumers.isEmpty() && this.consumers.isEmpty()) {
            for (String path : this.messageConsumers.keySet()) {
                if (!this.consumers.containsKey(path) && this.session != null) {
                    this.consumers.put(path, this.queueConsumer(path));
                }
                for (Connector.MessageConsumer mConsumer : this.messageConsumers.get(path)) {
                    Consumer<javax.jms.Message> messageConsumer = m -> mConsumer.accept(MessageReader.textFrom(m), this.callback((javax.jms.Message)m));
                    this.jmsMessageConsumers.put(mConsumer, messageConsumer.hashCode());
                    this.consumers.get(path).listen(messageConsumer);
                }
            }
        }
    }

    private synchronized void recoverEventsAndMessages() {
        this.recoverEvents();
        this.recoverMessages();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void recoverEvents() {
        if (this.eventOutBox == null) {
            return;
        }
        EventOutBox eventOutBox = this.eventOutBox;
        synchronized (eventOutBox) {
            if (!this.eventOutBox.isEmpty()) {
                while (!this.eventOutBox.isEmpty()) {
                    Map.Entry<String, Event> event = this.eventOutBox.get();
                    if (event == null) continue;
                    if (!this.doSendEvent(event.getKey(), event.getValue())) break;
                    this.eventOutBox.pop();
                }
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void recoverMessages() {
        if (this.messageOutBox == null) {
            return;
        }
        MessageOutBox messageOutBox = this.messageOutBox;
        synchronized (messageOutBox) {
            if (!this.messageOutBox.isEmpty()) {
                while (!this.messageOutBox.isEmpty()) {
                    Map.Entry<String, String> message = this.messageOutBox.get();
                    if (message == null) continue;
                    if (!this.doSendMessage(message.getKey(), message.getValue())) break;
                    this.messageOutBox.pop();
                }
            }
        }
    }

    private void checkConnection() {
        if (this.session != null && this.brokerUrl.startsWith("failover") && !this.connected.get()) {
            Logger.debug("Data-hub currently disconnected. Waiting for reconnection...");
            return;
        }
        if (this.connection != null && ((ActiveMQConnection)this.connection).isStarted() && this.session != null && ((ActiveMQSession)this.session).isRunning()) {
            this.connected.set(true);
            return;
        }
        Logger.debug("Restarting data-hub connection...");
        this.stop();
        try {
            this.connect();
        }
        catch (JMSException jMSException) {
            // empty catch block
        }
        this.connected.set(true);
    }

    private void initConnection() {
        try {
            this.connection = BusConnector.createConnection(this.brokerUrl, this.user, this.password, this.connectionListener());
            if (this.connection != null) {
                if (this.clientId != null && !this.clientId.isEmpty()) {
                    this.connection.setClientID(this.clientId);
                }
                this.connection.start();
            }
        }
        catch (JMSException e) {
            Logger.error(e);
        }
    }

    private String callback(javax.jms.Message m) {
        try {
            ActiveMQDestination replyTo = (ActiveMQDestination)m.getJMSReplyTo();
            return replyTo == null ? null : replyTo.getPhysicalName();
        }
        catch (JMSException e) {
            return null;
        }
    }

    private javax.jms.Message serialize(String payload, String replyTo) throws IOException, JMSException {
        javax.jms.Message message = MessageWriter.write(payload);
        message.setJMSReplyTo(this.session.createQueue(replyTo));
        message.setJMSCorrelationID(JmsConnector.createRandomString());
        return message;
    }

    private static javax.jms.Message serialize(String payload) throws IOException, JMSException {
        return MessageWriter.write(payload);
    }

    private static String createRandomString() {
        Random random = new Random(System.currentTimeMillis());
        long randomLong = random.nextLong();
        return Long.toHexString(randomLong);
    }

    private static javax.jms.Message serialize(Event event) throws IOException, JMSException {
        ByteArrayOutputStream os = new ByteArrayOutputStream();
        io.intino.alexandria.message.MessageWriter messageWriter = new io.intino.alexandria.message.MessageWriter(os);
        messageWriter.write(event.toMessage());
        messageWriter.close();
        return JmsConnector.serialize(os.toString());
    }

    private static class MessageDeserializer {
        private MessageDeserializer() {
        }

        static Message deserialize(javax.jms.Message message) {
            return new io.intino.alexandria.message.MessageReader(MessageReader.textFrom(message)).next();
        }
    }
}

