package io.intino.amidas.connectors.rocketchat;

import com.github.baloise.rocketchatrestclient.RocketChatClient;
import com.rocketchat.common.data.model.ErrorObject;
import com.rocketchat.common.data.model.Room;
import com.rocketchat.common.data.model.UserObject;
import com.rocketchat.common.listener.ConnectListener;
import com.rocketchat.common.listener.SubscribeListener;
import com.rocketchat.common.network.ReconnectionStrategy;
import com.rocketchat.core.RocketChatAPI;
import com.rocketchat.core.callback.FileListener;
import com.rocketchat.core.callback.LoginListener;
import com.rocketchat.core.factory.ChatRoomFactory;
import com.rocketchat.core.model.FileObject;
import com.rocketchat.core.model.RocketChatMessage;
import com.rocketchat.core.model.TokenObject;
import io.intino.alexandria.Json;
import io.intino.alexandria.cli.schemas.BotResponse;
import io.intino.alexandria.cli.schemas.BotTalk;
import io.intino.alexandria.logger.Logger;
import io.intino.alexandria.restaccessor.Response;
import io.intino.alexandria.restaccessor.core.RestAccessor;
import io.intino.alexandria.restaccessor.exceptions.RestfulFailure;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.*;

import static com.rocketchat.common.data.model.UserObject.Status.ONLINE;
import static io.intino.alexandria.cli.schemas.BotResponse.Type.*;
import static io.intino.amidas.connectors.rocketchat.ConfigurationKeys.*;
import static java.lang.String.format;

public class ProxyBot implements ConnectListener, LoginListener {
	private final Properties configuration;
	private final RocketChatClient chatClient;
	private final Set<String> subscribedRooms = new HashSet<>();
	private final Map<String, String> userTimeZones;
	private RocketChatAPI client;
	private String userId;

	public ProxyBot(Properties configuration, RocketChatClient chatClient) {
		this.configuration = configuration;
		this.chatClient = chatClient;
		this.userTimeZones = new HashMap<>();
	}

	public void start() {
		client = new RocketChatAPI(configuration.getProperty(Url));
		client.setReconnectionStrategy(new ReconnectionStrategy(20, 2000));
		client.connect(this);
	}

	@Override
	public void onConnect(String sessionID) {
		client.login(configuration.getProperty(BotUser), configuration.getProperty(BotPassword), this);
		client.setStatus(ONLINE, (aBoolean, errorObject) -> System.out.println(errorObject.getError()));
	}

	@Override
	public void onLogin(TokenObject token, ErrorObject error) {
		Logger.info(error == null ? configuration.getProperty(BotUser) + " connected successfully to Rocket.Chat" : "Got error " + error.getMessage());
		userId = token.getUserId();
		reloadSubscriptions();
	}

	private void reloadSubscriptions() {
		client.getRooms((rooms, e) -> {
			ChatRoomFactory factory = client.getChatRoomFactory();
			rooms.stream().map(ChatRoom::new).filter(r -> !isPublic(r)).forEach(factory::addChatRoom);
			factory.getChatRooms().forEach(r -> {
				if (!subscribedRooms.add(r.getRoomData().getRoomId())) return;
				r.subscribeRoomMessageEvent(emptySubscriber(), (roomId, message) -> new Thread(() -> {
					if (message.getSender().getUserId().equals(userId)) return;
					processMessage(factory.getChatRoomById(roomId), message);
				}).start());
			});
		});
	}

	private boolean isPublic(ChatRoom r) {
		return r.getRoomName().equalsIgnoreCase("GENERAL") || !r.type().equals(ChatRoom.Type.ONE_TO_ONE);
	}

	private void processMessage(RocketChatAPI.ChatRoom chatRoom, RocketChatMessage chatMessage) {
		String userId = chatMessage.getSender().getUserId();
		String message = chatMessage.getMessage();
		try {
			BotResponse response = queryApp(userId, message);
			if (response.type().equals(text) || response.type().equals(question)) sendTextResponse(chatRoom, response);
			else if (response.type().equals(multiline)) sendMultilineResponse(chatRoom, response);
			else if (response.type().equals(file)) sendFile(chatRoom, userId, response);
		} catch (RestfulFailure e) {
			Logger.error(e.getMessage());
			chatRoom.sendMessage("Error processing request: " + e.getMessage());
		}
	}

	private void sendTextResponse(RocketChatAPI.ChatRoom context, BotResponse botResponse) {
		context.sendMessage(format(botResponse.raw()));
	}

	private void sendMultilineResponse(RocketChatAPI.ChatRoom context, BotResponse botResponse) {
		context.sendMessage(format(botResponse.raw()));
	}

	private BotResponse queryApp(String userId, String message) throws RestfulFailure {
		io.intino.alexandria.restaccessor.RestAccessor.RestfulSecureConnection restAccessor = new RestAccessor().secure(urlOf(configuration.getProperty(AppUrl)), userId);
		BotTalk talk = new BotTalk().conversation(message).timeZone(timeZone(userId));
		Response post = restAccessor.post(configuration.getProperty(AppCliPath), Json.toJson(talk));
		return Json.fromJson(post.content(), BotResponse.class);
	}

	private URL urlOf(String url) {
		try {
			return new URI(url).toURL();
		} catch (MalformedURLException | URISyntaxException e) {
			Logger.error(e);
			return null;
		}
	}

	private void sendFile(RocketChatAPI.ChatRoom chatRoom, String userId, BotResponse response) {
		try {

			Path tempFile = Files.createTempFile(chatRoom.getRoomData().getRoomId() + "_" + userId + "_", response.fileName());
			Files.write(tempFile, io.intino.alexandria.Base64.decode(response.raw()));
			chatRoom.uploadFile(tempFile.toFile(), response.fileName(), response.title(), fileListener(tempFile));
		} catch (IOException e) {
			Logger.error(e);
		}
	}

	private String timeZone(String userId) {
		if (userTimeZones.containsKey(userId)) return this.userTimeZones.get(userId);
		try {
			com.github.baloise.rocketchatrestclient.model.User info = chatClient.getUsersApi().getInfo(userId);
			userTimeZones.put(userId, timeZone(info.getUtcOffset()));
			return userTimeZones.get(userId);
		} catch (IOException e) {
			return "Atlantic/Canary";
		}
	}

	private String timeZone(Integer offset) {
		if (offset == 0) return "Z";
		return (offset > 0 ? "+" + format("%02d", offset) : format("%03d", offset)) + ":00";
	}

	private SubscribeListener emptySubscriber() {
		return (isSubscribed, subId) -> {
		};
	}

	@Override
	public void onDisconnect(boolean closedByServer) {
		Logger.warn("Disconnected. By server ->" + closedByServer);
	}

	@Override
	public void onConnectError(Exception ex) {
		Logger.error(ex);
		System.exit(-1);
	}


	private FileListener fileListener(Path tempFile) {
		return new FileListener() {
			@Override
			public void onUploadStarted(String s, String s1, String s2) {

			}

			@Override
			public void onUploadProgress(int i, String s, String s1, String s2) {

			}

			@Override
			public void onUploadComplete(int i, FileObject fileObject, String s, String s1, String s2) {
				tempFile.toFile().delete();
			}

			@Override
			public void onUploadError(ErrorObject errorObject, IOException e) {
				tempFile.toFile().delete();
			}

			@Override
			public void onSendFile(RocketChatMessage rocketChatMessage, ErrorObject errorObject) {

			}
		};
	}

	public static class ChatRoom extends Room {


		public enum Type {
			PUBLIC,
			PRIVATE,
			ONE_TO_ONE;
		}

		private final Room object;

		public ChatRoom(Room object) {
			super(cover(object));
			this.object = object;
		}

		@Override
		public String getRoomName() {
			return object.getRoomName().isEmpty() ? getRoomId() : object.getRoomName();
		}

		@Override
		public String getRoomId() {
			return object.getRoomId();
		}

		public Type type() {
			return Type.valueOf(((Object) object.getRoomType()).toString());
		}

		@Override
		public UserObject getUserInfo() {
			return object.getUserInfo();
		}

		private static JSONObject cover(Room object) {
			try {

				return new JSONObject().put("_id", object.getRoomId()).put("t", "c");
			} catch (JSONException e) {
				return new JSONObject();
			}
		}
	}
}