package io.intino.amidas.connectors.rocketchat;

import com.github.baloise.rocketchatrestclient.RocketChatClient;
import com.github.baloise.rocketchatrestclient.model.Message;
import com.github.baloise.rocketchatrestclient.model.Room;
import com.github.baloise.rocketchatrestclient.model.User;
import io.intino.alexandria.logger.Logger;
import io.intino.amidas.shared.Team;
import io.intino.amidas.shared.connectors.MessagingConnector;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

public class RocketChatConnector implements MessagingConnector {

	private final RocketChatClient rc;
	private final List<User> users;
	private final ProxyBot proxyBot;

	public RocketChatConnector(Configuration configuration) {
		proxyBot = new ProxyBot(configuration);
		this.rc = new RocketChatClient(configuration.rocketChatUrl().toString(), configuration.botUser(), configuration.botPassword());
		this.users = new ArrayList<>();
		loadUsers();
	}


	@Override
	public boolean sendMessage(String userName, String message) {
		if (this.users.isEmpty()) loadUsers();
		User user = findUserByName(userName);
		if (user == null) return false;
		sendToUser(user, message);
		return true;
	}

	@Override
	public boolean sendMessage(String team, String channel, String message) {
		if (this.users.isEmpty()) loadUsers();
		User user = findUserByName(channel);
		if (user == null) return false;
		sendToUser(user, message);
		return true;
	}

	@Override
	public void setup(io.intino.alexandria.http.AlexandriaSpark alexandriaSpark) {

	}

	@Override
	public void start(Team team) {
		proxyBot.start();
	}

	private void sendToUser(User user, String message) {
		final Room room = roomOf(user);
		try {
			rc.getChatApi().postMessage(room, new Message(message));
		} catch (IOException e) {
			Logger.error(e.getMessage() + ":" + room.getRoomId() + " -> " + message);
		}
	}

	public User findUserByName(String aUser) {
		String user = aUser.startsWith("@") ? aUser.substring(1) : aUser;
		return users.stream().filter(u -> u.getUsername().equals(user)).findFirst().orElse(null);
	}

	private void loadUsers() {
		try {
			rc.getChannelsApi().list();
			Collections.addAll(this.users, rc.getUsersApi().list());
		} catch (Throwable e) {
			Logger.error(e.getMessage());
		}
	}

	private Room roomOf(User user) {
		Room room = new Room(user.getUsername(), true);
		room.setId("@" + user.getUsername());
		return room;
	}

}
